import json
import logging
import subprocess
from datetime import date

from sandbox import common
from sandbox import sdk2
from sandbox.common.types.misc import NotExists
from sandbox.common.types import task as ctt
from sandbox.projects.common.teamcity import TeamcityServiceMessagesLog
from sandbox.projects.sdc.common import arc
from sandbox.projects.sdc.common import constants
from sandbox.projects.sdc.common import git
from sandbox.projects.sdc.common import binaries_api
from sandbox.projects.sdc.common.constants import SANDBOX_RESOURCE_ATTRIBUTE
from sandbox.projects.sdc.common.constants import DEFAULT_BB_SERVER_URL, DEFAULT_REPO_URL
from sandbox.projects.sdc.common.constants import ROBOT_SDC_CI_SECRET_ID
from sandbox.projects.sdc.common.vcs_service import VcsService

from sandbox.projects.sdc.common import utils
from sandbox.projects.sdc.common.yt_helpers import create_clusters, check_yt_path_existence, YT_VERSION
from sandbox.projects.sdc.resource_types import SdcJobLayerYtPath
from sandbox.projects.sdc.SdcBuildBinariesPortoLayer import SdcBuildBinariesPortoLayer
from sandbox.projects.sdc.SdcBuildPackagesPortoLayer import SdcBuildPackagesPortoLayer
from sandbox.projects.sdc.SdcDeployPorto import SdcDeployPorto
from sandbox.projects.sdc.resource_types import (
    SdcBinariesLayerLogs, SdcLxcContainer, SdcSyncPackagesLayerLogs, SdcPortoLayersCommit
)
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2 import yav

log = logging.getLogger(__name__)


class SdcBuildAllPortoLayers(sdk2.Task):
    JOB_LAYER_YT_PATH_JSON = 'sdc_job_layer_yt_path.json'
    PORTO_LAYERS_COMMIT_FILE = 'porto_layers_commit.txt'

    class Requirements(sdk2.Requirements):
        environments = (
            PipEnvironment('yandex-yt', YT_VERSION),
        )

    class Parameters(sdk2.Parameters):
        branch_or_commit = sdk2.parameters.String('Branch or commit for building porto layers')
        timestamp = sdk2.parameters.Integer('Timestamp', required=False)
        clusters = sdk2.parameters.List('Clusters to upload layers', default=['hahn', 'arnold'])
        ctest_disabled = sdk2.parameters.Bool('Run build without ctests', default=False)
        build_sdc_tc_build_id = sdk2.parameters.Integer('Build SDC teamcity build id for reuse for binaries', required=False)
        build_sdc_sandbox_task_id = sdk2.parameters.Integer('Build SDC sandbox build id to reuse for binaries', required=False)
        repo_url = sdk2.parameters.String('SDC repo url', default=DEFAULT_REPO_URL)
        bb_server_url = sdk2.parameters.String('Bitbucket server base url', default=DEFAULT_BB_SERVER_URL)
        sdc_package = sdk2.parameters.String('base sdc package', default='sdc')
        stripped = sdk2.parameters.Bool('Return YT path to stripped binaries', default=False)
        bazel_target = sdk2.parameters.String('Bazel target for building light porto layers')
        py3_only = sdk2.parameters.Bool('[Deprecated] Build py3 only binaries', default=True)

        vcs_type = sdk2.parameters.String('VCS type', default=constants.DEFAULT_VCS_TYPE)
        arc_mode = sdk2.parameters.String('Arc mode', default=constants.DEFAULT_ARC_MODE)

        with sdk2.parameters.Group('Teamcity parameters'):
            publish_teamcity_logs = sdk2.parameters.Bool('Publish teamcity logs', default=False)
            commit = sdk2.parameters.String('Teamcity commit', hint=True)
            branch = sdk2.parameters.String('Teamcity branch')

        # we need container, because it has newer version of git, than default sandbox task
        container = sdk2.parameters.Container('LXC container with Docker', resource_type=SdcLxcContainer)

        with sdk2.parameters.Output(reset_on_restart=True):
            porto_layers = sdk2.parameters.List("Resulting porto layers")

    def on_save(self):
        if not self.Parameters.container:
            self.Parameters.container = SdcLxcContainer.find(attrs={"released": "stable"}).first().id

    def on_prepare(self):
        self.logger = logging.getLogger(self.__class__.__name__)
        secrets = yav.Yav(
            robot_sdc_ci=yav.Secret(ROBOT_SDC_CI_SECRET_ID)
            # do not forget to delegate new tokens
        )
        self.ssh_key = secrets.robot_sdc_ci['ssh']
        self.bb_token = secrets.robot_sdc_ci['token.bb']
        self.arc_token = secrets.robot_sdc_ci['token.arc']
        self.arcanum_token = secrets.robot_sdc_ci['token.arcanum']
        self.yt_token = secrets.robot_sdc_ci['token.yt']
        self.clusters = create_clusters(self.Parameters.clusters, self.yt_token)

        self.arc_cli = arc.build_client(self.arc_token)
        self.vcs_service = VcsService(vcs_type=self.Parameters.vcs_type,
                                      arc_token=self.arc_token,
                                      arcanum_token=self.arcanum_token,
                                      bb_token=self.bb_token,
                                      bb_server_url=self.Parameters.bb_server_url)

    def get_vcs_repo_dir(self):
        return self.path('arcadia') if self.is_arc() else self.path('sdc')

    def get_working_dir(self):
        repo_dir = self.get_vcs_repo_dir()
        return repo_dir.joinpath('sdg/sdc') if self.is_arc() else repo_dir

    def is_arc(self):
        return self.Parameters.vcs_type == constants.VCS_ARC

    def get_commit_timestamp(self, working_dir):
        with sdk2.helpers.ProcessLog(self, logger=log) as pl:
            if self.is_arc():
                commit_timestamp_cmd = ['arc', 'log', '-n', '1', '--format={date}']
            else:
                commit_timestamp_cmd = ['git', 'show', '-s', '--format=%cI', 'HEAD']
            process_output = subprocess.check_output(commit_timestamp_cmd,
                                                     stderr=pl.stdout,
                                                     cwd=working_dir)
            return process_output.decode().strip()

    def create_job_layer_yt_path(self, layers, commit, commit_timestamp, version, branch):
        job_layer_yt_path = {
            'job-layer-yt-path': layers,
            'commit': commit,
            'commit_timestamp': commit_timestamp,
        }
        if version:
            job_layer_yt_path['version'] = version
        if branch:
            job_layer_yt_path['branch'] = branch
        with open(self.JOB_LAYER_YT_PATH_JSON, 'w') as f:
            json.dump(job_layer_yt_path, f)
        job_layer_yt_path_resource = SdcJobLayerYtPath(
            self, 'YT path to porto layers', self.JOB_LAYER_YT_PATH_JSON
        )
        sdk2.ResourceData(job_layer_yt_path_resource).ready()

    def create_porto_layers_commit_resource(self):
        porto_layers_commit_resource = SdcPortoLayersCommit(
            self, 'Commit for which porto layers were built', self.PORTO_LAYERS_COMMIT_FILE
        )
        with open(self.PORTO_LAYERS_COMMIT_FILE, 'w') as f:
            f.write(self.Context.commit)
        sdk2.ResourceData(porto_layers_commit_resource).ready()

    def check_for_success(self, task_id):
        if sdk2.Task[task_id].status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure('Error in task {}'.format(task_id))

    def get_layer_resource_id(self, yt_path):
        layer_info = None
        for cluster in self.clusters:
            if not layer_info:
                try:
                    # do not follow symlinks, since we can't set attribute to cached ubuntu layer
                    # it will work for ordinary files
                    layer_info = cluster.client.get(yt_path + '&', attributes=[SANDBOX_RESOURCE_ATTRIBUTE])
                    break
                except self.cluster_unavailable_exceptions as e:
                    # ignore connection error due to cluster unavailability,
                    # assuming, that only one cluster can be unavailable at the time
                    logging.info('Cluster {} is unavailable'.format(cluster.name))
                    logging.exception(e)
        return layer_info.attributes[SANDBOX_RESOURCE_ATTRIBUTE]

    def publish_teamcity_logs(self):
        teamcity_log = str(self.path('teamcity.log'))
        with open(teamcity_log, 'w') as f:
            f.write(
                "##teamcity[setParameter name='{}' value='{}']".
                format('porto_layers', ','.join(self.Context.layers))
            )
        sdk2.ResourceData(
            TeamcityServiceMessagesLog(self, 'TeamCity log', teamcity_log)
        ).ready()

    def download_and_publish_important_logs(self, task_id, resource_type, desc, local_path):
        important_logs_resource = sdk2.Resource.find(
            task_id=task_id, type='SDC_IMPORTANT_LOGS'
        ).first()
        if important_logs_resource:
            path = self.agentr.resource_sync(important_logs_resource.id)
            new_resource = resource_type(self, desc, local_path)
            with open(str(new_resource.path), 'w') as new, open(str(path), 'r') as old:
                new.writelines(old.readlines())
            sdk2.ResourceData(new_resource).ready()

    def resolve_branch_and_commit(self):
        branch_or_commit = str(self.Parameters.branch_or_commit)
        if branch_or_commit:
            # for nirvana
            _, branch, commit = self.vcs_service.resolve_branch_and_commit(branch_or_commit)
            logging.info('branch={} commit={}'.format(branch, commit))
        else:
            # for teamcity
            branch, commit = self.Parameters.branch, self.Parameters.commit
        return branch, commit

    def resolve_vcs_branch_info(self, working_dir, commit):
        if self.is_arc():
            branch, detached = arc.get_branch_info(self.arc_cli, working_dir)
            logging.info('arc info branch: %s, detached: %s', branch, detached)
            return branch
        else:
            return git.get_branch_name(self, working_dir, self.ssh_key, commit)

    def update_latest_compatible_link(self):
        if self.Context.building_light_porto_layer:
            return
        # todo: partially move this logic to infra.sdcbinaries?
        base_layer_tasks = [self.Context.sdc_base_packages_task_id, self.Context.sdc_sync_packages_task_id]
        base_layers_rebuilt = any([t for t in base_layer_tasks if t is not NotExists])
        binaries_layer = self.Context.layers[0]
        latest_compatible_binaries = self.Context.latest_compatible_binaries
        if base_layers_rebuilt or self.is_midnight_dev() or not check_yt_path_existence(self.clusters, latest_compatible_binaries):
            logging.info('Trying to create symlink {} for latest compatible binaries {}'.format(
                latest_compatible_binaries, binaries_layer
            ))
            for cluster in self.clusters:
                try:
                    cluster.client.link(binaries_layer, latest_compatible_binaries, force=True)
                except self.cluster_unavailable_exceptions as e:
                    # ignore connection error due to cluster unavailability,
                    # assuming, that only one cluster can be unavailable at the time
                    logging.info('Cluster {} is unavailable'.format(cluster.name))
                    logging.exception(e)

    def is_midnight_dev(self):
        vcs_midnight_branches = ['midnight-dev', 'tags/groups/sdg/sdc/midnight-dev']
        return self.Context.branch in vcs_midnight_branches

    def arc_checkout(self, repo_dir, branch, commit):
        arc_mode = self.Parameters.arc_mode
        logging.info('arc checkout, mode: %s', arc_mode)
        try:
            rev = commit or branch
            obj_store_dir = str(self.path('object_store'))
            logging.info('arc dir: %s, object store dir: %s', repo_dir, obj_store_dir)
            return arc.sdc_checkout(self.arc_token, arc_mode, rev, repo_dir, obj_store_dir)
        except Exception:
            logging.exception('Arc checkout exception occurs')
            raise common.errors.TaskFailure('Arc checkout failed')

    def vcs_checkout(self, repo_dir, branch, commit):
        if self.is_arc():
            arc.initialize_arc_credentials(self.arc_token)
            return self.arc_checkout(repo_dir, branch, commit)
        else:
            git.initialize_git()
            git.git_cached_checkout(self, self.Parameters.repo_url, repo_dir, self.ssh_key, branch, commit)

    def _get_version(self):
        branch = self.Context.branch
        stable_tag_prefix = 'tags/groups/sdg/pipeline/car/'
        rc_prefix = 'releases/sdg/pipeline/car-rc/'
        release_prefix = 'releases/sdg/pipeline/car/'
        version = None

        if self.is_midnight_dev():
            version = 'midnight-dev-{}'.format(date.today().strftime('%Y.%m.%d'))
        elif branch.startswith('{}master'.format(stable_tag_prefix)):
            short_name = branch[len(stable_tag_prefix):]
            version = '{}-{}'.format(short_name, date.today().strftime('%Y.%m.%d'))
        elif branch.startswith(rc_prefix):
            version = 'prestable-{}'.format(date.today().strftime('%Y.%m.%d'))
        elif branch.startswith(release_prefix):
            version = 'release-{}'.format(date.today().strftime('%Y.%m.%d'))
        elif branch.startswith(('release', 'master', 'prestable')) \
                and not branch.startswith('releases/'):
            version = '{}-{}'.format(self.Context.branch, date.today().strftime('%Y.%m.%d'))

        return version

    def build_full_porto_layers(self):
        with self.memoize_stage.build_sdc_binaries:
            yt_path = self.Context.layers[0]
            if not check_yt_path_existence(self.clusters, yt_path):
                sdc_binaries_task = SdcBuildBinariesPortoLayer(
                    self, branch=self.Context.branch, commit=self.Context.commit, yt_path=yt_path,
                    clusters=self.Parameters.clusters, ctest_disabled=self.Parameters.ctest_disabled,
                    build_sdc_tc_build_id=self.Parameters.build_sdc_tc_build_id,
                    build_sdc_sandbox_task_id=self.Parameters.build_sdc_sandbox_task_id,
                    bb_server_url=self.Parameters.bb_server_url,
                    vcs_type=self.Parameters.vcs_type,
                )
                self.Context.sdc_binaries_task_id = sdc_binaries_task.enqueue().id

        with self.memoize_stage.build_sdc_base_packages:
            yt_path = self.Context.layers[2]
            if not check_yt_path_existence(self.clusters, yt_path):
                sdc_base_packages_task = SdcBuildPackagesPortoLayer(
                    self, package_type='base', package=self.Parameters.sdc_package,
                    branch=self.Context.branch, commit=self.Context.commit,
                    parent_layer_resource_id=self.Context.ubuntu_resource_id, yt_path=yt_path,
                    clusters=self.Parameters.clusters,
                    bb_server_url=self.Parameters.bb_server_url,
                    vcs_type=self.Parameters.vcs_type,
                )
                self.Context.sdc_base_packages_task_id = sdc_base_packages_task.enqueue().id
                raise sdk2.WaitTask(
                    [self.Context.sdc_base_packages_task_id],
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
                )

        with self.memoize_stage.get_sdc_base_packages_resource_id:
            yt_path = self.Context.layers[2]
            if self.Context.sdc_base_packages_task_id:
                self.check_for_success(self.Context.sdc_base_packages_task_id)
                self.Context.base_packages_resource_id = sdk2.Resource.find(
                    task_id=self.Context.sdc_base_packages_task_id,
                    type='PORTO_LAYER_SDC'
                ).first().id
            else:
                self.Context.base_packages_resource_id = self.get_layer_resource_id(yt_path)

        with self.memoize_stage.build_sdc_sync_packages:
            yt_path = self.Context.layers[1]
            if not check_yt_path_existence(self.clusters, yt_path):
                sdc_sync_packages_task = SdcBuildPackagesPortoLayer(
                    self, package_type='sync', package=self.Parameters.sdc_package,
                    branch=self.Context.branch, commit=self.Context.commit,
                    parent_layer_resource_id=self.Context.base_packages_resource_id, yt_path=yt_path,
                    clusters=self.Parameters.clusters,
                    bb_server_url=self.Parameters.bb_server_url,
                    vcs_type=self.Parameters.vcs_type,
                )
                self.Context.sdc_sync_packages_task_id = sdc_sync_packages_task.enqueue().id

    def build_light_porto_layer(self):
        with self.memoize_stage.sdc_deploy_light_porto:
            yt_path = self.Context.layers[0]
            if not check_yt_path_existence(self.clusters, yt_path):
                sdc_deploy_light_porto_task = SdcDeployPorto(
                    self, branch=self.Context.branch, commit=self.Context.commit,
                    bazel_target=self.Parameters.bazel_target, clusters=self.Parameters.clusters,
                    bb_server_url=self.Parameters.bb_server_url, vcs_type=self.Parameters.vcs_type,
                )
                self.Context.sdc_deploy_light_porto_task_id = sdc_deploy_light_porto_task.enqueue().id
            else:
                logging.info('Light porto layer with the same path already exists: {}'.format(yt_path))

    def on_execute(self):
        with utils.create_venv() as venv:
            utils.configure_env(venv)
            with self.memoize_stage.compute_layers:
                repo_dir = str(self.get_vcs_repo_dir())
                working_dir = str(self.get_working_dir())
                branch, commit = self.resolve_branch_and_commit()
                mount_point = self.vcs_checkout(repo_dir, branch, commit)  # noqa: auto unmount at the end of scope: on_execute()
                porto_layers = binaries_api.get_porto_layers(working_dir, self.Parameters.stripped, self.Parameters.bazel_target)
                self.Context.layers = porto_layers['layers']
                self.Context.latest_compatible_binaries = porto_layers['latest_compatible_binaries_layer']
                self.Context.building_light_porto_layer = True if self.Parameters.bazel_target else False
                self.Context.branch = utils.normalize_branch(branch)
                self.Context.commit = commit
                self.Context.commit_timestamp = self.get_commit_timestamp(working_dir)
                self.Context.version = self._get_version()

            with self.memoize_stage.build_base_ubuntu:
                yt_path = self.Context.layers[1] if self.Context.building_light_porto_layer else self.Context.layers[3]
                if not check_yt_path_existence(self.clusters, yt_path):
                    raise common.errors.TaskFailure(
                        'Base ubuntu isn\'t present in all clusters {}'.format(self.Parameters.clusters)
                    )
                self.Context.ubuntu_resource_id = self.get_layer_resource_id(yt_path)

            if self.Context.building_light_porto_layer:
                self.build_light_porto_layer()
            else:
                self.build_full_porto_layers()

            with self.memoize_stage.wait_for_completion:
                tasks_to_wait = [
                    self.Context.sdc_binaries_task_id, self.Context.sdc_sync_packages_task_id,
                    self.Context.sdc_deploy_light_porto_task_id
                ]
                self.Context.tasks_to_wait = [t for t in tasks_to_wait if t is not NotExists]
                if self.Context.tasks_to_wait:
                    raise sdk2.WaitTask(
                        self.Context.tasks_to_wait, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
                    )

            with self.memoize_stage.publish_logs:
                if self.Context.building_light_porto_layer:
                    if self.Context.sdc_deploy_light_porto_task_id:
                        important_logs = SdcBinariesLayerLogs(self, 'Sdc binaries layer log', 'sdc_binaries_layer_logs.log')
                        with open(str(important_logs.path), 'w') as f:
                            f.write(
                                'Sandbox task: https://sandbox.yandex-team.ru/task/{}/view'.format(
                                    self.Context.sdc_deploy_light_porto_task_id
                                )
                            )
                        sdk2.ResourceData(important_logs).ready()
                else:
                    if self.Context.sdc_binaries_task_id:
                        self.download_and_publish_important_logs(
                            self.Context.sdc_binaries_task_id, SdcBinariesLayerLogs,
                            'Sdc binaries layer logs', 'sdc_binaries_layer_logs.log'
                        )

                    if self.Context.sdc_sync_packages_task_id:
                        self.download_and_publish_important_logs(
                            self.Context.sdc_sync_packages_task_id, SdcSyncPackagesLayerLogs,
                            'Sdc sync packages layer logs', 'sdc_sync_packages_layer_logs.log'
                        )

                self.create_porto_layers_commit_resource()

            with self.memoize_stage.check_for_success:
                for task_id in self.Context.tasks_to_wait:
                    self.check_for_success(task_id)

            with self.memoize_stage.publish_output:
                self.update_latest_compatible_link()
                self.create_job_layer_yt_path(
                    self.Context.layers, self.Context.commit, self.Context.commit_timestamp,
                    self.Context.version, self.Context.branch
                )

                if self.Parameters.publish_teamcity_logs:
                    self.publish_teamcity_logs()

                self.Parameters.porto_layers = self.Context.layers
