import logging
import os
import textwrap
import time

from sandbox import sdk2
from sandbox.projects.sandbox.sandbox_lxc_image import SandboxLxcImage, UbuntuRelease
from sandbox.projects.sdc.common.constants import (
    ROBOT_SDC_CI_SECRET_ID,
)
from sandbox.projects.sdc.resource_types import SdcLxcContainerWithDocker
from sandbox.projects.sdc.SdcBuildLxcContainerWithDocker.constants import (
    APT_SOURCES,
    DOCKER_GPG,
    DOCKER_START,
    DOCKER_STOP,
    ENVIRONMENT,
    FUSE,
    IPTABLES_RULES,
    SUDOERS,
)

import sandbox.common.types.misc as ctm


INSTALL_PACKAGES = '''set -ve

apt-get install --yes --no-install-recommends yandex-internal-root-ca
apt-get --yes purge yandex-search-common-apt

apt-key add /docker.gpg

apt-get update

apt-get install --yes --no-install-recommends \\
    atop \\
    curl \\
    fuse \\
    squashfs-tools \\
    pigz \\
    python3-urllib3=1.22-1ubuntu0.18.04.2 \\
    sdc-pbuilder \\
    debmake \\
    python-setuptools \\
    python-pygit2 \\
    yandex-yt-python \\
    python-virtualenv \\
    python-psycopg2 \\
    python-jinja2 \\
    python-enum34 \\
    python-botocore \\
    python-boto3 \\
    python-six \\
    python-requests \\
    python-psutil \\
    python-numpy \\
    python3-six \\
    python3-ruamel.yaml \\
    python3-requests \\
    python3-boto3 \\
    python3-jinja2 \\
    python3-retrying \\
    git \\
    python-pip \\
    software-properties-common \\
    apt-transport-https \\
    bc \\
    libffi-dev \\
    docker-ce

usermod -aG docker sandbox
usermod -aG docker zomb-sandbox
'''


DOCKER_PULL = '''set -ve
docker info
docker login -u sandbox -p "$DOCKER_REGISTRY_TOKEN" registry.yandex.net
docker pull registry.yandex.net/sdc/yandexsdc-deb:{docker_tag}
docker tag registry.yandex.net/sdc/yandexsdc-deb:{docker_tag} yandexsdc-deb:{docker_tag}
docker image list
'''


class SdcBuildLxcContainerWithDocker(SandboxLxcImage):

    IMAGE_SIZE_LIMIT = 30 * 1024

    class Requirements(SandboxLxcImage.Requirements):
        cores = 17  # 1 is ok, but > 16 is required to prevent docker failures on multislot
        dns = ctm.DnsType.DNS64
        disk_space = 64 * 1024

    class Parameters(SandboxLxcImage.Parameters):
        # Sandbox ENV settings
        docker_tag = sdk2.parameters.String('docker_tag', default='2022-07-08-12-13')

        resource_type = SandboxLxcImage.Parameters.resource_type(
            default=SdcLxcContainerWithDocker.name,
        )

        test_result_lxc = SandboxLxcImage.Parameters.test_result_lxc(default=False)
        custom_image = SandboxLxcImage.Parameters.custom_image(default=True)
        ubuntu_release = SandboxLxcImage.Parameters.ubuntu_release(default=UbuntuRelease.BIONIC)

        custom_env_secrets = SandboxLxcImage.Parameters.custom_env_secrets(
            default={
                'DOCKER_REGISTRY_TOKEN': ROBOT_SDC_CI_SECRET_ID + '#token.registry',
            },
        )

    def _write_rootfs_file(
        self,
        filepath,
        contents,
        append=False,
        mode='644',
        user=None,
        group=None,
    ):
        rootfs_path = os.path.join(self.rootfs, filepath.lstrip('/'))
        rootfs_dirname = os.path.dirname(rootfs_path)

        logging.info(
            'Writing file %s (%s)\n%s\n%s\n%s',
            rootfs_path,
            filepath,
            '=' * 100,
            contents,
            '=' * 100,
        )
        try:
            os.makedirs(rootfs_dirname)
        except OSError:
            pass

        writemode = 'w'
        if append:
            writemode = 'a'
        with open(rootfs_path, '{writemode}b'.format(writemode=writemode)) as fp:
            fp.write(contents.encode())
        self.execute(
            'chmod {mode} {filepath}'.format(mode=mode, filepath=filepath),
            chroot=True
        )

        if user is not None:
            self.execute(
                'chown {user} {filepath}'.format(user=user, filepath=filepath),
                chroot=True
            )
        if group is not None:
            self.execute(
                'chgrp {group} {filepath}'.format(group=group, filepath=filepath),
                chroot=True
            )

    def _make_docker_bootstrap_image(self):
        commands = []
        commands.append('mkdir -p "$1"/docker/overlay2/l')
        commands.append('cp -rp /var/lib/docker/image "$1"/docker/')

        for filename in os.listdir(os.path.join(self.rootfs, 'var/lib/docker/overlay2')):
            if filename == 'l':
                continue
            commands.append(
                'ln -s /var/lib/docker/overlay2/{filename}'
                ' "$1"/docker/overlay2/{filename}'
                ''.format(
                    filename=filename,
                )
            )

        for filename in os.listdir(
            os.path.join(self.rootfs, 'var/lib/docker/overlay2/l')
        ):
            if filename == 'l':
                continue
            commands.append(
                'ln -s /var/lib/docker/overlay2/l/{filename}'
                ' "$1"/docker/overlay2/l/{filename}'
                ''.format(
                    filename=filename,
                )
            )

        return commands

    def cook_image(self, current_dir):
        result = super(SdcBuildLxcContainerWithDocker, self).cook_image(current_dir)

        self._write_rootfs_file(
            '/etc/apt/sources.list.d/sdc-sources.list',
            APT_SOURCES,
            append=True,
        )
        self._write_rootfs_file('/docker.gpg', DOCKER_GPG)
        self._write_rootfs_file('/etc/environment', ENVIRONMENT)
        self._write_rootfs_file('/etc/iptables/rules.v6', IPTABLES_RULES)

        logging.info(self.execute_custom_script(INSTALL_PACKAGES))

        self._write_rootfs_file('/etc/sudoers.d/sandbox', SUDOERS, mode='440')
        self._write_rootfs_file('/etc/fuse.conf', FUSE)

        self.execute(
            'PATH="$PATH:{rootfs}/bin:{rootfs}/usr/bin"'
            ' dockerd'
            ' --data-root {rootfs}/var/lib/docker'
            ' --exec-root {rootfs}/var/run/docker'
            ' --host unix://{rootfs}/var/run/docker.sock'
            ' --pidfile {rootfs}/var/run/docker.pid &'.format(
                rootfs=self.rootfs.rstrip('/'),
            )
        )

        # wait for docker daemon to start
        time.sleep(5)
        self.execute('chmod 777 /var/run/docker.sock', chroot=True)

        logging.info(self.execute_custom_script(DOCKER_PULL.format(
            docker_tag=self.Parameters.docker_tag,
        )))

        self.execute(
            'kill -9 `cat {rootfs}/var/run/docker.pid`'.format(
                rootfs=self.rootfs.rstrip('/'),
            )
        )
        time.sleep(10)

        # we will run docker our way
        self.execute('rm -rf /lib/systemd/system/docker.service', chroot=True)

        self._write_rootfs_file(
            '/usr/bin/sdc-start-docker',
            DOCKER_START,
            mode='755',
        )
        self._write_rootfs_file(
            '/usr/bin/sdc-stop-docker',
            DOCKER_STOP,
            mode='755',
        )

        self._write_rootfs_file(
            '/etc/sandbox/on_task_start',
            '#!/bin/bash\n/etc/sandbox/onstart "$1" >>"$1"/onstart.out 2>>"$1"/onstart.err',
            mode='755',
        )

        self._write_rootfs_file(
            '/etc/sandbox/onstart',
            textwrap.dedent('''
                #!/bin/bash
                set -x
                test -e /dev/kvm || mknod /dev/kvm c 10 232

                # start docker, it creates needed directories
                /usr/bin/sdc-start-docker "$1"
                # TODO check readiness better than sleep
                sleep 5

                # stop docker and change image contents
                /usr/bin/sdc-stop-docker
                {docker_bootstrap_image}

                # start docker again
                /usr/bin/sdc-start-docker "$1"
                # TODO check readiness better than sleep
                sleep 5
                chmod 777 /var/run/docker.sock
            ''').format(
                docker_bootstrap_image='\n'.join(self._make_docker_bootstrap_image()),
            ).strip(),
            mode='755',
        )
        self.cleanup_image()

        return result

    def cleanup_image(self):
        self.execute(
            'rm -rf /var/lib/apt/lists/* /root/.docker',
            chroot=True,
        )
