import six

import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.projects.porto import BuildPortoLayer
from sandbox.projects.sdc.PreparePackage import PreparePackage
from sandbox.projects.sdc.porto_helper import PortoLayerSdc
from sandbox.projects.sdc.porto_helper.AbstractSdcBuildPortoLayer import AbstractSdcBuildPortoLayer


class SdcBuildPackagesPortoLayer(AbstractSdcBuildPortoLayer):
    class Parameters(AbstractSdcBuildPortoLayer.Parameters):
        package_type = sdk2.parameters.String('Package type (base, full, sync)', required=True)
        package = sdk2.parameters.String('base sdc package', required=True)
        with sdk2.parameters.Group('Porto parameters') as porto_parameters:
            parent_layer_resource_id = sdk2.parameters.Integer('Parent layer resource id', required=True)
            layer_type = sdk2.parameters.String('Layer type', required=True, default='PORTO_LAYER_SDC')
            layer_name = sdk2.parameters.String('Layer name')
            compress = sdk2.parameters.String('Compress')
            script_url = sdk2.parameters.String(
                'Setup scripts URLs (arcadia:/path[@rev] or https://...)', multiline=True
            )
            script_env = sdk2.parameters.Dict('Environment variables')
            merge_all_layers = sdk2.parameters.Bool('Merge all layers', default=False)
            space_limit = sdk2.parameters.Integer('Max disk usage (MB)', default=30000)
            memory_limit = sdk2.parameters.Integer('Max memory usage (MB)', default=8192)
            debug_build = sdk2.parameters.Bool('Debug build', default=False)
            start_os = sdk2.parameters.Bool('Start OS for scripts', required=True, default=True)

    def on_enqueue(self):
        super(SdcBuildPackagesPortoLayer, self).on_enqueue()
        self.Context.layer_build_param_timeout = 5 * 60 * 60
        if self.Parameters.package_type == 'base':
            self.Context.ttl = 'inf'
        self.Context.script_url = (
            'arcadia:/robots/trunk/genconf/PORTOVM/sdc/get_sdc_base_porto_builder.sh\n'
            'arcadia:/robots/trunk/genconf/PORTOVM/common/cleanup.sh'
        )

    def vault_env(self):
        vault_env = dict()
        vault_env['BB_TOKEN'] = 'sdc-bb-token'
        vault_env['ARCANUM_TOKEN'] = 'sdc-arcanum-token'
        return vault_env

    def script_env(self):
        script_env = dict(self.Parameters.script_env)
        script_env['REVISION'] = self.Parameters.commit
        script_env['BB_SERVER_URL'] = self.Parameters.bb_server_url
        if self.Parameters.package_type == 'sync':
            script_env['BUILD_SYNC_LAYER'] = 1
        script_env['SDC_VERSION_SUFFIX'] = sdk2.Task[self.Context.preparation_task_id].Context.package_version
        script_env['SDC_PACKAGE'] = self.Parameters.package
        script_env['VCS_TYPE'] = self.Parameters.vcs_type
        return script_env

    def create_preparation_task(self, sdc_important_logs_resource):
        # todo: get rid of branch_or_commit in the future,
        #  when I remove legacy tasks for porto layers building
        branch_or_commit = ''
        if self.Parameters.branch:
            branch_or_commit += self.Parameters.branch + ','
        branch_or_commit += self.Parameters.commit
        return PreparePackage(
            self, branch_or_commit=branch_or_commit,
            important_logs_parent_resource=sdc_important_logs_resource,
            bb_server_url=self.Parameters.bb_server_url,
            vcs_type=self.Parameters.vcs_type,
        )

    def get_porto_layer_params(self):
        return {
            BuildPortoLayer.ParentLayer.name: self.Parameters.parent_layer_resource_id,
            BuildPortoLayer.LayerType.name: self.Parameters.layer_type,
            BuildPortoLayer.LayerName.name: self.Context.layer_name,
            BuildPortoLayer.Compress.name: self.Context.compress,
            BuildPortoLayer.ScriptUrl.name: self.Context.script_url,
            BuildPortoLayer.StartOS.name: self.Parameters.start_os,
            BuildPortoLayer.MergeLayers.name: self.Parameters.merge_all_layers,
            BuildPortoLayer.DebugBuild.name: self.Parameters.debug_build,
            BuildPortoLayer.SpaceLimit.name: self.Parameters.space_limit,
            BuildPortoLayer.MemoryLimit.name: self.Parameters.memory_limit,
            BuildPortoLayer.ScriptEnv.name: self.script_env(),
            BuildPortoLayer.VaultEnv.name: self.vault_env(),
            BuildPortoLayer.OutputResourceID.name: self.Context.porto_layer_resource_id,
            'kill_timeout': self.Context.layer_build_param_timeout
        }

    def build_porto_layer(self):
        with self.memoize_stage.build_porto_layer:
            sdc_porto_layer_resource = PortoLayerSdc(
                self, 'Porto layer', self.get_porto_layer_filename(), ttl=self.Context.ttl
            )
            self.Context.porto_layer_resource_id = sdc_porto_layer_resource.id
            layer_build_params = self.get_porto_layer_params()
            build_layer_task_class = sdk2.Task['BUILD_PORTO_LAYER']
            layer_build_task = build_layer_task_class(
                self,
                description=self.Parameters.description,
                **{
                    key: value.id if isinstance(value, sdk2.Resource) else value
                    for key, value in six.iteritems(layer_build_params)
                }
            ).enqueue()
            self.Context.build_layer_task_id = layer_build_task.id
            raise sdk2.WaitTask(
                [self.Context.build_layer_task_id],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True
            )

        with self.memoize_stage.upload_to_yt:
            self.check_for_success(self.Context.build_layer_task_id)
            # we created resource, but we still have to download it to current machine
            sdk2.ResourceData(
                sdk2.Resource.find(task_id=self.id, type=self.Parameters.layer_type).first()
            )
            self.upload_layer_to_yt(
                self.get_porto_layer_filename(), self.clusters, self.Parameters.commit,
                self.Parameters.branch, self.Context.porto_layer_resource_id
            )
