import logging
import threading

import sandbox.common.types.client as ctc

from sandbox import common
from sandbox import sdk2
from sandbox.sdk2.os import enable_turbo_boost, disable_turbo_boost
from sandbox.projects.sdc.SdcBuild import SdcBuild

logger = logging.getLogger(__name__)


class SdcBuildPerfCtestsE5(SdcBuild):
    class Requirements(SdcBuild.Requirements):
        client_tags = ctc.Tag.INTEL_E5_2650V2
        disk_space = 120 * 1024  # 120 GB

    class Parameters(SdcBuild.Parameters):
        vcs_steps_path = sdk2.parameters.String('Path to 0.sh script to Run',
                                                default='tc_build/build_sdc_perf')
        raise_custom_metrics_upload_failures = sdk2.parameters.Bool('Raise custom metrics failure upload errors',
                                                                    default=False)
        kill_timeout = 3 * 3600  # 3 hour

    def get_default_env_variables(self):
        envs = super(SdcBuild, self).get_default_env_variables()
        strict_upload_errors = self.Parameters.raise_custom_metrics_upload_failures
        envs['T__RAISE_CUSTOM_METRICS_UPLOAD_ERRORS'] = '1' if strict_upload_errors else '0'
        return envs

    def get_turbo_boost_off(self):
        with open('/sys/devices/system/cpu/intel_pstate/no_turbo', 'r') as f:
            return int(f.readline().rstrip())

    def check_turbo_boost(self):
        if self.get_turbo_boost_off():
            return
        else:
            raise common.errors.TaskFailure('Turbo Boost is enabled!')

    def log_turbo_boost(self):
        threading.Timer(5.0, self.log_turbo_boost).start()
        logger.info("Content of /sys/devices/system/cpu/intel_pstate/no_turbo is {}".format(
            self.get_turbo_boost_off()
        ))

    def on_execute(self):
        try:
            disable_turbo_boost()
            self.check_turbo_boost()
            self.log_turbo_boost()
            super(SdcBuild, self).on_execute()
        except Exception as e:
            raise e
        finally:
            enable_turbo_boost()
