import errno
import datetime
import logging
import os

from sandbox import sdk2
from sandbox.projects.sdc.common import utils
from sandbox.projects.sdc.common.constants import ROBOT_SDC_CI_SECRET_ID
from sandbox.projects.sdc.common_tasks.BaseSdcTask import BaseSdcTask
from sandbox.projects.sdc.SdcLanesMapValidatorRelease import SdcLanesMapValidatorRelease
from sandbox.projects.sdc.resource_types import SdcLanesMapValidatorBinaries

import sandbox.common.types.client as ctc

log = logging.getLogger(__name__)


class SdcLanesMapValidatorBuild(BaseSdcTask):
    sdc_deploy_dst = 'lanes_map_validator.tgz'

    class Requirements(BaseSdcTask.Requirements):
        cores = 32
        ram = 32 * 1024  # 32GB
        disk_space = 30 * 1024  # 30 GB
        client_tags = ctc.Tag.SSD

    class Parameters(BaseSdcTask.Parameters):
        enable_release = sdk2.parameters.Bool(
            'Release built resource. Disabled checks build itself only.', hint=True)

    def get_buildsdcript_env_variables(self):
        envs = super(SdcLanesMapValidatorBuild, self).get_buildsdcript_env_variables()
        envs.update({'T__SDC_DEPLOY_DST': self.sdc_deploy_dst})
        return envs

    def sdc_deploy_path(self):
        """Path to sdc_deploy()ed artifact"""
        return os.path.join(self.get_yandexsdc_path(), 'sdc_deploy', self.sdc_deploy_dst)

    def on_build_script_success(self):
        if self.Parameters.enable_release:
            resource = SdcLanesMapValidatorBinaries(
                self, 'LanesMapValidator binaries archive', self.sdc_deploy_path(),
                ttl=30, branch=utils.normalize_branch(self.branch), commit=self.commit)
            sdk2.ResourceData(resource).ready()
        super(SdcLanesMapValidatorBuild, self).on_build_script_success()

    def get_yp_oauth_token(self):
        return sdk2.yav.Secret(ROBOT_SDC_CI_SECRET_ID).data()['token.yp']

    def on_success(self, prev_status):
        if self.Parameters.enable_release:
            lanes_server_release = SdcLanesMapValidatorRelease(self, task_id_to_release=self.id)
            lanes_server_release.enqueue()
        super(SdcLanesMapValidatorBuild, self).on_success(prev_status)

    @property
    def release_template(self):
        subject = 'Lanes Map Validator RESOURCE released!'
        message = 'Release date: {}'.format(datetime.datetime.now())
        default_emails = [
            'super-sk@yandex-team.ru',
            'sergeynorin@yandex-team.ru',
            'lu4ik@yandex-team.ru'
        ]
        release_types = ['stable']
        return sdk2.task.ReleaseTemplate(default_emails, subject, message, release_types)
