import datetime
import logging
import os
import tarfile

from sandbox import sdk2
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.sdc.common import utils
from sandbox.projects.sdc.common.constants import ROBOT_SDC_CI_SECRET_ID
from sandbox.projects.sdc.common_tasks.BaseSdcTask import BaseSdcTask
from sandbox.projects.sdc.SdcLanesRoutingServerRelease import SdcLanesRoutingServerRelease
from sandbox.projects.sdc.resource_types import SdcLanesRoutingServerBinaries

import sandbox.common.types.client as ctc

log = logging.getLogger(__name__)


class SdcLanesRoutingServerBuild(release_integration.ReleaseToYaDeployTask2, BaseSdcTask):
    class Requirements(BaseSdcTask.Requirements):
        cores = 32
        ram = 32 * 1024  # 32GB
        disk_space = 30 * 1024  # 30 GB
        client_tags = ctc.Tag.SSD

    class Parameters(BaseSdcTask.Parameters):
        binaries_output_dir = sdk2.parameters.String(
            'Lanes Routing Server build output binaries dir', hint=True)
        release_resource = sdk2.parameters.Bool(
            'Release builded RESOURCE to deploy', hint=True)

    def on_build_script_success(self):
        path_to_binaries_dir = self.path(self.Parameters.binaries_output_dir)
        path_to_binaries_tgz = self.path('server_binaries.tgz')
        logging.info('Going to pack dir {} to RESOURCE binary {}'.format(
            path_to_binaries_dir, path_to_binaries_tgz))
        self.gzip_directory(str(path_to_binaries_dir), str(path_to_binaries_tgz))

        resource = SdcLanesRoutingServerBinaries(
            self, 'LanesRoutingServer binaries archive', path_to_binaries_tgz,
            ttl=30, branch=utils.normalize_branch(self.branch), commit=self.commit)
        sdk2.ResourceData(resource).ready()
        super(SdcLanesRoutingServerBuild, self).on_build_script_success()

    def gzip_directory(self, source, destination):
        with tarfile.open(destination, "w:gz") as fh:
            for item in os.listdir(source):
                fh.add(os.path.join(source, item), arcname=item)

    def get_yp_oauth_token(self):
        return sdk2.yav.Secret(ROBOT_SDC_CI_SECRET_ID).data()['token.yp']

    def on_success(self, prev_status):
        if self.Parameters.release_resource:
            lanes_server_release = SdcLanesRoutingServerRelease(self, task_id_to_release=self.id)
            lanes_server_release.enqueue()
        super(SdcLanesRoutingServerBuild, self).on_success(prev_status)

    @property
    def release_template(self):
        subject = 'Lanes Routing Server RESOURCE released!'
        message = 'Release date: {}'.format(datetime.datetime.now())
        default_emails = ['amareelez@yandex-team.ru', 'ilzar@yandex-team.ru', 'shcheglovnd@yandex-team.ru']
        release_types = ['testing', 'stable']
        return sdk2.task.ReleaseTemplate(default_emails, subject, message, release_types)

    def on_release(self, additional_parameters):
        release_integration.ReleaseToYaDeployTask2.on_release(self, additional_parameters)
        super(SdcLanesRoutingServerBuild, self).on_release(additional_parameters)
