import datetime
import logging

from sandbox import common, sdk2
from sandbox.projects.common.vcs.arc import Arc, ArcCommandFailed
from sandbox.projects.sdc.common.arc import ARC_SDC_REL_PATH, initialize_arc_credentials
from sandbox.projects.sdc.common.bitbucket import BitbucketClient
from sandbox.projects.sdc.common_tasks.BaseSdcTask import BaseSdcTask

import sandbox.common.types.client as ctc

log = logging.getLogger(__name__)

MIDNIGHT_DEV_BRANCHES_PREFIX = 'tags/groups/sdg/sdc/midnight-dev'
MIDNIGHT_DEV_BRANCHE_NAME_TEMPLATE = MIDNIGHT_DEV_BRANCHES_PREFIX + '-%Y.%m.%d'
MIDNIGHT_DEV_TAG = 'tags/groups/sdg/sdc/midnight-dev'
MIDNIGHT_DEV_BRANCH_GIT = 'midnight-dev'

DEFAULT_ARC_LOG_WINDOW_LENGTH = 100
MAX_NUMBER_OF_ARC_LOG_GETTING = 5


class MidnightDevUpdaterException(common.errors.TaskFailure):
    pass


class SdcMidnightDevUpdater(BaseSdcTask):
    class Requirements(BaseSdcTask.Requirements):
        cores = 2
        ram = 4 * 1024  # 4GB
        disk_space = 30 * 1024  # 30 GB
        client_tags = ctc.Tag.SSD

    class Parameters(BaseSdcTask.Parameters):
        sync_with_git_branch = sdk2.parameters.Bool(
            'Update midnight-dev branch in in correspondence with midnight-dev branch from git.',
            default=False
        )

    def _delete_remote_tag(self, tag_name):
        self.arc_cli._execute_command(
            self.arc_mount_point,
            [self.arc_cli.binary_path, 'push', '-d', tag_name]
        )

    def create_new_midnight_dev_branch(self, fork_commit):
        branch_name = datetime.datetime.today().strftime(MIDNIGHT_DEV_BRANCHE_NAME_TEMPLATE)
        log.info(
            'Will create branch [%s] for new midnight-dev branch from commit [%s].',
            branch_name, fork_commit
        )

        self.arc_cli.fetch(mount_point=self.arc_mount_point, branch='trunk')
        self.arc_cli.checkout(
            mount_point=self.arc_mount_point,
            branch=branch_name,
            create_branch=True,
            start_point=fork_commit
        )

        try:
            self.arc_cli.push(mount_point=self.arc_mount_point, upstream=branch_name)
        except ArcCommandFailed:
            log.exception(
                '`arc push` newly created branch {} is failed! '
                'Probably this branch was already created.'
            )
            raise

        log.info('Branch %s has been created and pushed to remote repo.', branch_name)
        return branch_name

    def change_midnight_dev_tag(self, new_midnight_dev_branch):
        log.info('Will move tag [%s] to HEAD of branch [%s].',
                 MIDNIGHT_DEV_TAG, new_midnight_dev_branch)

        self._delete_remote_tag(MIDNIGHT_DEV_TAG)
        log.info('Tag %s was deleted.', MIDNIGHT_DEV_TAG)

        self.arc_cli.push(
            mount_point=self.arc_mount_point,
            refspecs=[(new_midnight_dev_branch, MIDNIGHT_DEV_TAG)],
        )
        log.info('Tag [%s] was pointed at branch [%s].', MIDNIGHT_DEV_TAG, new_midnight_dev_branch)

    def get_corresponding_arc_commit_to_git_midnight_dev(self):
        bb_client = BitbucketClient(self.bb_token, self.Parameters.bb_server_url)
        base_git_commit = bb_client.get_commit_from_branch_or_commit(MIDNIGHT_DEV_BRANCH_GIT)

        arc_log_start_commit = None
        arc_log_window_length = DEFAULT_ARC_LOG_WINDOW_LENGTH
        for i in range(MAX_NUMBER_OF_ARC_LOG_GETTING):
            log.info(
                'Get new chunk of history from arc log by vcs path: [%s].'
                'Iteration #%d; arc log start commit: %s, window size: %s',
                ARC_SDC_REL_PATH,
                i,
                arc_log_start_commit or 'HEAD', arc_log_window_length
            )

            arc_commit_log = self.arc_cli.log(
                mount_point=self.arc_mount_point,
                path=ARC_SDC_REL_PATH,
                max_count=arc_log_window_length,
                start_commit=arc_log_start_commit,
                as_dict=True
            )

            for commit_info in arc_commit_log:
                sync_commit_hash = commit_info.get('attributes', {}).get('sync.commit')
                if sync_commit_hash == base_git_commit:
                    log.info(
                        'Found corresponding arc commit to git base commit for midnight dev.'
                        'git base commit: %s; arc commit: %s',
                        base_git_commit,
                        commit_info.get('commit')
                    )
                    return commit_info.get('commit')

            # do not use .get() here cause if we use default value None of '' - arc log will
            # every iteration give you log starting from HEAD. Better fall down.
            arc_log_start_commit = arc_commit_log[-1]['commit']
            arc_log_window_length *= 2
        else:
            log.exception(
                'Failed to find corresponding arc commit to git base commit for midnight dev.'
            )
            raise MidnightDevUpdaterException(
                'Failed to find corresponding arc commit to git base commit for midnight dev.'
                'git base commit: {}; arc last checked commit: {}'.format(
                    base_git_commit, arc_commit_log[-1].get("commit", ""))
            )

    def get_base_commit_for_new_midnight_dev_branch(self):
        if self.Parameters.sync_with_git_branch:
            return self.get_corresponding_arc_commit_to_git_midnight_dev()
        last_commit_for_sdc = self.arc_cli.log(
            mount_point=self.arc_mount_point,
            path='sdg/sdc',
            max_count=1,
            as_dict=True
        )
        last_commit = last_commit_for_sdc[0].get('commit')

        if not last_commit:
            log.exception(
                'Failed to find last arc commit for path `sdg/sdc`.'
                'arc log returned %s',
                last_commit_for_sdc
            )
            raise MidnightDevUpdaterException(
                'Failed to find last arc commit for path `sdg/sdc`.'
            )

        return last_commit

    def on_prepare(self):
        super(SdcMidnightDevUpdater, self).on_prepare()
        self.arc_cli = Arc()

    def on_execute(self):
        mount_point = self.vcs_checkout()  # noqa: auto unmount at the end of scope: on_execute()
        self.arc_mount_point = mount_point._mount_point
        initialize_arc_credentials(self.arc_token)

        commit_hash = self.get_base_commit_for_new_midnight_dev_branch()
        new_branch = self.create_new_midnight_dev_branch(commit_hash)
        self.change_midnight_dev_tag(new_branch)
