import base64

from sandbox import common
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.helpers import process

from sandbox.projects.sdc.common import aws
from sandbox.projects.sdc.common import yt_helpers
from sandbox.projects.sdc.common_tasks.BaseSdcTask import BaseSdcTask
from sandbox.projects.sdc.common import utils
from sandbox.projects.sdc.common import arc
from sandbox.projects.sdc.resource_types import SdcLxcContainerCI
import sandbox.common.types.task as ctt


class SdcPrCompositeLxc(BaseSdcTask):
    class Requirements(BaseSdcTask.Requirements):
        cores = 1
        ram = 2 * 1024  # 2GB
        disk_space = 10 * 1024  # 10 GB

        # https://sandbox.yandex-team.ru/admin/semaphores/50208191/view
        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='SDC_PR_COMPOSITE_MAX_INFLIGHT')
            ]
        )

    class Parameters(BaseSdcTask.Parameters):
        vcs_steps_path = sdk2.parameters.String('Path to 0.sh script to Run',
                                                default='misc/pr_composite')

        post_run_vcs_steps_path = sdk2.parameters.String('Path to 0.sh script to Run',
                                                         default='misc/pr_composite_post_run')

        container = sdk2.parameters.Container('LXC container with CI env',
                                              resource_type=SdcLxcContainerCI)

        kill_timeout = 60 * 60 * 10  # 10 hour

    def on_save(self):
        if self.Parameters.container:
            return

        container_search_attrs = {"released": "stable"}
        self.Parameters.container = SdcLxcContainerCI.find(attrs=container_search_attrs).first().id

    def on_prepare(self):
        super(SdcPrCompositeLxc, self).on_prepare()
        venv = utils.create_venv()
        utils.configure_env(venv)
        utils.configure_network()

        arc.initialize_arc_credentials(self.arc_token)
        aws.initialize_aws_credentials(self.aws_key_id, self.aws_secret_key)
        yt_helpers.initialize_yt_credentials(self.yt_token)

        self._vcs_repo = self.vcs_checkout()  # self._vcs_repo prevent auto-unmount at the end of scope
        self.set_vcs_resulting_parameters()

    def on_terminate(self):
        self.logger.info('on_terminate called')
        self.do_post_run_stage()

    def on_finish(self, prev_status, status):
        self.logger.info('on_finish called: %s => %s', prev_status, status)
        self.do_post_run_stage()

    def on_before_timeout(self, seconds):
        if seconds <= 60:
            self.logger.info('on_before_timeout(%s) => call post_run script', seconds)
            self.do_post_run_stage()

    def do_post_run_stage(self):
        with self.memoize_stage.post_run_stage:
            try:
                self.post_run_impl()
            except Exception:
                self.logger.exception('Post run exception occurs')

    def post_run_impl(self):
        self.logger.info('Evaluate post_run script:')

        working_dir = self.get_working_dir()
        cmd_to_call = self.get_buildscript_subprocess_cmd(steps_paths=self.Parameters.post_run_vcs_steps_path)
        with process.ProcessLog(self, logger="post_run") as pl:
            sp.call(cmd_to_call, cwd=str(working_dir), stdout=pl.stdout, stderr=pl.stdout)

        # Collect & Upload important artifacts
        self.Parameters.runtime_json_output = self.get_runtime_json_output()
        uploaded_resources = self.upload_resources()
        if uploaded_resources:
            self.on_resources_ready(uploaded_resources)

        # Do info reporting
        info = self.make_info_string()
        self.set_info(info, do_escape=False)

    def on_execute(self):
        self.logger.info('Task {} started'.format(self.__class__.__name__))

        ret_code = self.run_build_script()

        is_success = ret_code == 0

        if not is_success:
            failure_msg = 'CI scripts finished with non zero exit code({})'.format(ret_code)
            raise common.errors.TaskFailure(failure_msg)

        self.logger.info('Task {} finished'.format(self.__class__.__name__))

    def make_info_string(self):
        info_parts = []
        # Show resource urls
        task_resources = sdk2.Resource.find(task_id=self.id).limit(0)
        if task_resources:
            info_parts.append('<h3>Resources</h3><hr>')

        for resource in task_resources:
            type = resource.type
            path = resource.path
            url = resource.http_proxy
            row = '<a href="{}">{}</a>'.format(url, '{} -> {}'.format(type, path))
            info_parts.append(row)

        if task_resources:
            info_parts.append('<hr>')

        # Show build problems
        problems = self.Parameters.runtime_build_problems
        if problems:
            info_parts.append('<h3>Build problems</h3>')
            for build_problem in problems:
                message = build_problem.get('message')
                info_parts.append('<b style="color:red">{}</b>'.format(message))

        # join with <br> except <hr> tag
        auto_br_parts = []
        prev_el = None
        for p in info_parts:
            if prev_el and not prev_el.endswith('<hr>'):
                auto_br_parts.append('<br>')
            auto_br_parts.append(p)
            prev_el = p

        info = ''.join(auto_br_parts)
        return info

    def get_default_env_variables(self):
        envs = super(SdcPrCompositeLxc, self).get_default_env_variables()
        envs.update(self.pr_composite_on_lxc_envs())
        envs.update(self.get_isolate_cloud_default_envs())
        return envs

    def pr_composite_on_lxc_envs(self):
        return {
            'LXC_MODE': '1',
            # Some scripts (infra/docker) read ~/.docker/config.json file to obtain registry auth
            # Create special env variable for this case
            'DOCKER_REGISTRY_AUTH': self.get_docker_registry_auth()
        }

    def get_isolate_cloud_default_envs(self):
        tu = 'https://isolate-cloud.sdc.yandex-team.ru/workers-api/triton-instances?required_worker_capabilities=ci'
        force_tests = [
            'testbackward.*',
            'testforward.*',
            'test_extractors_pipeline',
            'testcurve.*',
            'testsine.*',
            'testturn.*',
            'testmpc.*',
            'testring.*',
            'distributed_map.*',
            'pycore_pytests',
            'test_bl_motion_trajectory',
            'test_bl_simulation',
            'test_paired_tools',
            'test_tracker_metrics',
            'test_ypt_camera_processing.*',
            'test_ypt_multirow_scene_filter',
            'test_ypt_traffic_lights_filters.*',
            'testcommand_speed_smoothness',
            'verify_ride_from_yt_test'
        ]
        force_tests_regexp = '^(?!{}).*$'.format('|'.join(force_tests))
        isolate_cloud_envs = {
            'T__ISOLATE_CLOUD_AUTOENV_LOGBROKER_TIMEOUT': '300',
            'T__ISOLATE_CLOUD_GPU_SUPPORT': 'enabled(empty_to_disable)',
            'T__ISOLATE_CLOUD_PACK_MIN_SIZE': '3',
            'T__ISOLATE_CLOUD_PACK_PREF_SIZE': '3',
            'T__ISOLATE_CLOUD_PACK_WEIGHT_LIMIT': '1500',
            'T__ISOLATE_CLOUD_RUN_SIM_ONLY': '',
            'T__ISOLATE_CLOUD_SERVER_URL': 'https://isolate-cloud.sdc.yandex-team.ru',
            'T__ISOLATE_CLOUD_TASK_PRIORITY': '',
            'T__TRITON_INSTANCES_URL': tu,
            'T__ISOLATE_CLOUD_FORCE_TESTS_REGEXP': force_tests_regexp,
            # For now: any valid OAUTH2 token is required
            'T__SECRET__ISOLATE_CLOUD_TOKEN': str(self.teamcity_token)
        }

        return isolate_cloud_envs

    def get_docker_registry_auth(self):
        username = self.Parameters.username
        token = self.registry_token

        str2encode = '{}:{}'.format(username, token)
        # https://wiki.yandex-team.ru/docker-registry/#eslijetonesrabotalo
        return base64.b64encode(str2encode.encode())

    @property
    def artifacts_with_direct_link(self):
        r = super(SdcPrCompositeLxc, self).artifacts_with_direct_link
        patterns = set()
        patterns.update(r)
        patterns.add('**/pr_builds.html')
        patterns.add('**/triggered_builds.html')
        patterns.add('**/triggered_tests.html')
        return patterns
