import logging
import os
import tarfile


from datetime import datetime
from sandbox import sdk2
from sandbox.projects.sdc.common.constants import ROBOT_SDC_CI_SECRET_ID
from sandbox.projects.sdc.common.yt_helpers import create_clusters, YT_VERSION
from sandbox.projects.sdc.resource_types import SdcResourcesArchive, SdcLxcContainer
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2 import yav


log = logging.getLogger(__name__)


class SdcUploadResourcesArchiveToYT(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = (
            PipEnvironment('yandex-yt', YT_VERSION),
        )

    def on_prepare(self):
        secrets = yav.Yav(
            robot_sdc_ci=yav.Secret(ROBOT_SDC_CI_SECRET_ID),
            # do not forget to delegate new tokens
        )
        self.yt_token = secrets.robot_sdc_ci['token.yt']

    class Parameters(sdk2.Parameters):
        squashfs_with_resources = sdk2.parameters.Resource(
            'Squashfs with resources', resource_type=SdcResourcesArchive
        )
        with_test_resources = sdk2.parameters.Bool('Upload to YT layer with test resources', default=False)
        update_latest_symlink = sdk2.parameters.Bool('Update symlink to latest resources layer', default=False)
        container = sdk2.parameters.Container('LXC container with Docker', resource_type=SdcLxcContainer)

    def on_save(self):
        if not self.Parameters.container:
            self.Parameters.container = SdcLxcContainer.find(attrs={"released": "stable"}).first().id
        if not self.Parameters.squashfs_with_resources:
            resources_archive = SdcResourcesArchive.find(
                attrs={
                    "with_test_resources": self.Parameters.with_test_resources,
                    "released": "stable",
                }
            ).first()
            if resources_archive:
                self.Parameters.squashfs_with_resources = resources_archive.id
            else:
                log.warning('SquashFS with resources wasnt found')

    def on_execute(self):
        yandexsdc_path = self.path('yandexsdc')
        path = sdk2.ResourceData(self.Parameters.squashfs_with_resources).mount()
        sdk2.os.mount_overlay(yandexsdc_path, [str(path)], str(self.path('upper_dir')), str(self.path('work_dir')))
        resources_dirs = [
            os.path.join(str(yandexsdc_path), 'cached_files'),
            os.path.join(str(yandexsdc_path), 'cached_resources'),
            os.path.join(str(yandexsdc_path), 'cached_resources_symlinks'),
        ]
        date = datetime.now().strftime('%Y_%m_%d')
        sdc_resources_filename = 'sdc_resources_{}.tar'.format(date)
        sdc_resources_tar_archive = str(os.path.join(str(self.path()), sdc_resources_filename))
        with tarfile.open(sdc_resources_tar_archive, 'w') as tar:
            for resources_dir in resources_dirs:
                if os.path.exists(resources_dir):
                    arcname = os.path.join('/opt/yandexsdc', os.path.relpath(resources_dir, str(yandexsdc_path)))
                    tar.add(resources_dir, arcname=arcname)
        self.call(['pigz', sdc_resources_tar_archive])
        compressed_tar_archive = sdc_resources_tar_archive + '.gz'
        cluster = create_clusters(['hahn'], self.yt_token)[0]
        sdc_resources_layer_yt_path = '//home/selfdriving/porto_layers/{}'.format(sdc_resources_filename + '.gz')
        with open(compressed_tar_archive) as f:
            cluster.client.write_file(sdc_resources_layer_yt_path, f)
        if self.Parameters.update_latest_symlink:
            latest_resources_yt_path = '//home/selfdriving/porto_layers/sdc_resources_latest.tar.gz'
            cluster.client.link(sdc_resources_layer_yt_path, latest_resources_yt_path, force=True)
        sdk2.os.umount(yandexsdc_path)

    def call(self, cmd, cwd=os.getcwd()):
        with sdk2.helpers.ProcessLog(self, logger=log) as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT, cwd=cwd)
