import logging
import tempfile
import json
import time
import os
from enum import Enum
from datetime import datetime

from sandbox import common, sdk2
from sandbox.projects.sdc.common.aws import initialize_aws_credentials
from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.sdc.common.arc import initialize_arc_credentials, ARC_SDC_REL_PATH
from sandbox.projects.sdc.common_tasks.BaseSdcTask import BaseSdcTask

from .juggler_api_wrapper import create_status_event, report_events

log = logging.getLogger(__name__)


DEFAULT_CAR_BRANCHES_PREFIXES_TO_MONITOR = [
    'releases/sdg/pipeline/car/',
    'releases/sdg/pipeline/car-rc/',
    'tags/groups/sdg/pipeline/car/',
]

DEFAULT_ROBOT_BRANCHES_PREFIXES_TO_MONITOR = [
    'releases/sdg/pipeline/robot/',
    'tags/groups/sdg/pipeline/robot/',
]

DEFAULT_CAR_BRANCHES_TO_IGNORE = [
    'trunk',
    'releases/sdg/pipeline/car/2022-02-20-master-unmanned-sync-from-bb',
    'releases/sdg/pipeline/car/2022-02-20-master-sync-from-bb',
    'releases/sdg/pipeline/car/2022-02-20-master-lts-sync-from-bb',
    'releases/sdg/pipeline/car-rc/2022-02-11-SDC-92092',
    'releases/sdg/pipeline/car-rc/2022-02-22-SDC-94005',
    'releases/sdg/pipeline/car-rc/2022-03-04-SDC-94634',
    'releases/sdg/pipeline/car-rc/2022-03-16-SDC-95696',
    'releases/sdg/pipeline/car/2022-01-28-kazan-FDC-SDC-97225',
    'releases/sdg/pipeline/car-rc/2022-04-01-prestable-SDC-97423',
    'releases/sdg/pipeline/car/2022-04-19-fake-release-SDC-61525',
    'releases/sdg/pipeline/car/2022-04-15-technical-SDC-100776',
    'releases/sdg/pipeline/car-rc/2022-05-06-prestable-SDC-102322',
]

DEFAULT_ROBOT_BRANCHES_TO_IGNORE = [
    'trunk',
    'releases/sdg/pipeline/robot/2022-02-18-chromium-SDC-93133',
    'releases/sdg/pipeline/robot/2022-02-25-manganese-SDC-93900',
    'releases/sdg/pipeline/robot/2022-02-25-manganese-SDC-93900-calibration',
]

CAR_S3_PLATFORM_PREFIX = 'car'
ROBOT_S3_PLATFORM_PREFIX = 'sadr'
MAX_ARC_LOG_LENGTH = 10


class SdgDeployArtifactsMonitoringException(common.errors.TaskFailure):
    pass


class MonitoringStatus(Enum):
    def __str__(self):
        return str(self.name)
    OK = 1
    WARN = 2
    CRIT = 3


def date_to_timestamp(date_str):
    return datetime.strptime(date_str, '%Y-%m-%dT%H:%M:%S%z').timestamp()


class CommitInfo(object):
    def __init__(self, info_from_arc_log):
        self.commit = info_from_arc_log.get('commit')
        self.date = info_from_arc_log.get('date')

    def __str__(self):
        return '{' + '\'commit\': {}, \'date\': {}'.format(self.commit, self.date) + '}'


class SdgDeployArtifactsMonitoring(BaseSdcTask):
    class Requirements(BaseSdcTask.Requirements):
        cores = 2
        ram = 4 * 1024  # 4GB
        disk_space = 10 * 1024  # 10 GB

    class Parameters(BaseSdcTask.Parameters):
        car_branches_prefixes = sdk2.parameters.List(
            'Branches(also tags) to monitor for car platform.',
            default=DEFAULT_CAR_BRANCHES_PREFIXES_TO_MONITOR
        )
        robot_branches_prefixes = sdk2.parameters.List(
            'Branches(also tags) to monitor for robot(aka sadr) platform.',
            default=DEFAULT_ROBOT_BRANCHES_PREFIXES_TO_MONITOR
        )
        car_branches_prefixes_ignore = sdk2.parameters.List(
            'Branches(also tags) to ignore from monitoring for car platform.',
            default=DEFAULT_CAR_BRANCHES_TO_IGNORE,
        )
        robot_branches_prefixes_ignore = sdk2.parameters.List(
            'Branches(also tags) to ignore from monitoring for robot(aka sadr) platform.',
            default=DEFAULT_ROBOT_BRANCHES_TO_IGNORE,
        )
        monitoring_warn_threshold_sec = sdk2.parameters.Integer(
            'Monitoring WARN THRESHOLD',
            default=1800,  # 30 minutes
        )
        monitoring_crit_threshold_sec = sdk2.parameters.Integer(
            'Monitoring CRIT THRESHOLD',
            default=3600,  # 60 minutes
        )

    def get_last_commits_in_branch(self, branch):
        log_info = self.arc_cli.log(
            mount_point=self.arc_mount_point,
            branch=branch,
            path=ARC_SDC_REL_PATH,
            max_count=MAX_ARC_LOG_LENGTH,
            as_dict=True,
        )

        return [CommitInfo(log_item) for log_item in log_info]

    def get_last_artifact_meta_from_s3(self, platform,  branch):
        import botocore.exceptions

        version = platform + '/' + branch
        fn_meta = tempfile.mkstemp()[1]
        meta = {}
        try:
            self.s3mds_cas.download_meta(version, fn_meta)
            with open(fn_meta, 'r') as f:
                meta = json.load(f)
        except botocore.exceptions.ClientError as error:
            log.error(str(error))
        os.remove(fn_meta)
        return meta

    def fetch_all(self):
        for branch_prefixes in self.platform_to_branches_prefixes.values():
            for branch_prefix in branch_prefixes:
                self.arc_cli.fetch(self.arc_mount_point, branch_prefix)

    def list_all_branches(self):
        branches_and_tags = set()
        for branch_info in self.arc_cli.branch(self.arc_mount_point, all=True, as_dict=True):
            name = branch_info.get('name')
            if not name:
                raise RuntimeError('Branch info doesn\'t have a name attribute.')
            # as we fetch all of the branches, this condition always should be true
            if not branch_info.get('local', False) and name.startswith('arcadia/'):
                # prefix indicates that it is remote branch, we don't need it
                name = name[len('arcadia/'):]
            branches_and_tags.add(name)

        # decode() - because self.arc_cli.list_tags() returns bytes string
        for tag in self.arc_cli.list_tags(self.arc_mount_point).decode().split('\n'):
            branches_and_tags.add(tag)

        log.info('All branches and tags: %s', str(branches_and_tags))
        return branches_and_tags

    def create_monitoring_event(self, platform, branch, commits_info, deploy_artifacts_meta):
        artifacts_commit = deploy_artifacts_meta.get('commit')
        res_status = None
        res_description = 'branch \'{}\'; platform: \'{}\': '.format(branch, platform)
        if commits_info[0].commit == artifacts_commit:  # HEADs are equal
            res_status = MonitoringStatus.OK
            res_description += 'Deploy artifacts are up to date.'
        else:
            commits_without_artifacts = []
            for commit_info in commits_info:
                if commit_info.commit == artifacts_commit:
                    break
                commits_without_artifacts.append(commit_info)
            log.info('Commits without artifacts: %s', ','.join(map(str, commits_without_artifacts)))
            log.info('First commit without artifacts: %s', str(commits_without_artifacts[-1]))

            if len(commits_without_artifacts) == len(commits_info):
                res_status = MonitoringStatus.CRIT
                res_description += 'Cannot find last commit with built artifacts in arc log. ' \
                                   'Queue of SDC_DEPLOY_CAR tasks must be very big.'
            else:
                time_delta = int(
                    time.time() - date_to_timestamp(commits_without_artifacts[-1].date))
                res_description += 'Artifacts late for {} seconds.'.format(time_delta)
                if time_delta > self.Parameters.monitoring_crit_threshold_sec:
                    res_status = MonitoringStatus.CRIT
                elif time_delta > self.Parameters.monitoring_warn_threshold_sec:
                    res_status = MonitoringStatus.WARN
                else:
                    res_status = MonitoringStatus.OK

        assert res_status is not None and res_description is not None, \
            'Something went wrong with creation of status or description ' \
            'for branch \'{}\'; platform: \'{}\''.format(branch, platform)
        return create_status_event(platform, branch, res_status, res_description)

    def on_prepare(self):
        super(SdgDeployArtifactsMonitoring, self).on_prepare()
        self.arc_cli = Arc()

        self.platform_to_branches_prefixes = {
            CAR_S3_PLATFORM_PREFIX: self.Parameters.car_branches_prefixes,
            ROBOT_S3_PLATFORM_PREFIX: self.Parameters.robot_branches_prefixes,
        }

        self.platform_to_ignored_branches = {
            CAR_S3_PLATFORM_PREFIX: self.Parameters.car_branches_prefixes_ignore,
            ROBOT_S3_PLATFORM_PREFIX: self.Parameters.robot_branches_prefixes_ignore,
        }

    def on_execute(self):
        from core.infra.mds import casync

        mount_point = self.vcs_checkout()  # noqa: auto unmount at the end of scope: on_execute()
        self.arc_mount_point = mount_point._mount_point
        initialize_arc_credentials(self.arc_token)
        initialize_aws_credentials(self.aws_key_id, self.aws_secret_key)
        self.s3mds_cas = casync.cas_sdcbinaries(
            key_id=self.aws_key_id, key_secret=self.aws_secret_key)

        self.fetch_all()

        all_branches = self.list_all_branches()
        monitoring_events = []
        for branch in all_branches:
            # last_commit_date has format like "2022-04-25T13:44:48+03:00"
            commits_info = self.get_last_commits_in_branch(branch)
            for platform, branches_prefixes in self.platform_to_branches_prefixes.items():
                if branch in self.platform_to_ignored_branches.get(platform, []):
                    log.info('Ignore branch %s for platform %s', branch, platform)
                    continue

                for prefix in branches_prefixes:
                    if branch.startswith(prefix):
                        meta = self.get_last_artifact_meta_from_s3(platform, branch)
                        log.info('Create monitoring event for platform \'%s\'; branch \'%s\'; '
                                 'branch_prefix \'%s\';', platform, branch, branches_prefixes)
                        monitoring_events.append(
                            self.create_monitoring_event(platform, branch, commits_info, meta)
                        )

        report_events(monitoring_events)
