import json
import logging

from sandbox import common

MAX_RETRIES_SEND_EVENTS_TO_JUGGLER = 5
MAX_EVENTS_PER_BATCH = 100
API_URL = 'http://juggler-api.search.yandex.net'
PUSH_EVENT_URL = 'http://juggler-push.search.yandex.net:80/events'
EVENT_SOURCE = 'sdg-deploy-sandbox-artifacts-monitoring'
HOST = 'sdg-deploy-artifacts-readiness-check'

log = logging.getLogger(__name__)


class SdcDeployArtifactsMonitoringJugglerApiException(common.errors.TaskFailure):
    pass


def make_request_to_juggler(check_data):
    import requests

    session = requests.Session()
    response = session.request(
        'POST',
        PUSH_EVENT_URL,
        data=check_data,
        headers={'Content-Type': 'application/json'})
    if response.status_code != 200:
        message = 'response: {} ,method: {}, url: {}, data: {}'
        logging.debug(
            message.format(str(response.__dict__), 'POST', PUSH_EVENT_URL,
                           str(check_data)))
        response.raise_for_status()

    return response.json()


def create_status_event(platform, branch, status, description):
    service = '{}.{}'.format(platform, branch)
    return {
        'description': description,
        'host': HOST,
        'instance': '',
        'service': service,
        'status': str(status),
    }


def send_events(events):
    from retry.api import retry_call
    import requests

    status_events = {
        'source': EVENT_SOURCE,
        'events': events,
    }

    result = retry_call(
        make_request_to_juggler,
        fargs=[json.dumps(status_events)],
        exceptions=requests.exceptions.HTTPError,
        tries=5,
        delay=5,
    )
    logging.debug('POST {} to {}: {}'.format(json.dumps(status_events), PUSH_EVENT_URL, result))
    return result


def filter_events_by_response(events, response):
    new_events = []
    for event, response_for_event in zip(events, response.get('events')):
        if response_for_event.get('code') != 200:
            new_events.append(event)
    return new_events


def report_events_batch(events_batch):
    initial_events_size = len(events_batch)

    for i in range(MAX_RETRIES_SEND_EVENTS_TO_JUGGLER):
        log.info('report_events_batch, iteration #%d; events_batch lenght: %d',
                 i, len(events_batch))
        if len(events_batch) <= 0:  # nothing to send
            break

        response = send_events(events_batch)
        if not response.get('success', False):
            log.error('Failed to send events to juggler.')
            raise SdcDeployArtifactsMonitoringJugglerApiException(
                'Failed to send events to juggler.')

        num_of_accepted_events = response.get('accepted_events', 0)
        if num_of_accepted_events == len(events_batch):
            log.info('All events were successfully sent to juggler.')
            return

        log.info('Only %d/%d events were sent successfully, iteration #%d',
                 num_of_accepted_events, len(events_batch), i)
        events_batch = filter_events_by_response(events_batch, response)

    if len(events_batch) > 0:
        log.error('Failed to send some events to juggler. Number failed events to send: %d/%d',
                  len(events_batch), initial_events_size)
        raise SdcDeployArtifactsMonitoringJugglerApiException(
            'Failed to send some events to juggler.')


def report_events(events):
    log.info('report_events; initial size of events: %d', len(events))
    number_of_batches = len(events) // MAX_EVENTS_PER_BATCH + 1
    for i in range(number_of_batches):
        log.info('report_events; try to send batch #%d', i)
        report_events_batch(events[i * MAX_EVENTS_PER_BATCH: (i + 1) * MAX_EVENTS_PER_BATCH])
