import json
import logging
import os
import sys
import tempfile

from sandbox.common import fs
from sandbox.projects.common.decorators import retries
from sandbox.projects.common.vcs import arc as arc_cli
from sandbox.projects.sdc.common import constants
from sandbox.sdk2.helpers import process
from sandbox.sdk2.helpers import subprocess as sp

log = logging.getLogger(__name__)

ARC_SDC_REL_PATH = 'sdg/sdc'


def build_client(token):
    return arc_cli.Arc(arc_oauth_token=token)


def extract_commit_info(arc_token, mount_point):
    arc_cli = build_client(arc_token)
    top_commit = arc_cli.log(mount_point=mount_point, max_count=1, as_dict=True)[0]
    sdc_commit = arc_cli.log(mount_point=mount_point, max_count=1, as_dict=True, path='sdg/sdc')[0]
    return top_commit, sdc_commit


def initialize_arc_credentials(token):
    log.info('Initialize arc token')
    token_dir = os.path.join(os.path.expanduser('~'), '.arc')
    if not os.path.exists(token_dir):
        os.mkdir(token_dir)
    with open(os.path.join(token_dir, 'token'), 'w') as f:
        f.write(token)


@retries(max_tries=10, delay=60)
def sdc_checkout(token, mode, rev, working_dir, obj_store_path=None):
    """
       Checkout arc repo for SDC project with 2 modes: mount or export.
       Return mount_point (for context on mount) or None on export
    """

    cli = build_client(token)

    ctx_object = None  # mount point or arc bare repo context object
    if mode == constants.ARC_MODE_MOUNT:
        ctx_object = mount(cli, rev, working_dir, obj_store_path)
    else:
        init_bare(cli, working_dir)
        with tempfile.mkdtemp(dir='.') as arc_dir:
            with mount(cli, rev, arc_dir, obj_store_path):
                cli.export(arc_dir, rev, ARC_SDC_REL_PATH, working_dir)

    sdc_path = os.path.join(working_dir, ARC_SDC_REL_PATH)
    sys.path.insert(0, sdc_path)

    return ctx_object


def init_bare(cli, working_dir):
    cmd = [cli.binary_path, 'init', '--bare']

    fs.make_folder(working_dir)
    proc = process.subprocess.Popen(
        cmd,
        cwd=working_dir,
        stdout=process.subprocess.PIPE,
        stderr=process.subprocess.PIPE
    )
    out, err = proc.communicate()

    return_code = proc.poll()
    if return_code:
        raise RuntimeError('Cannot init arc repo in {} dir, out: {}, err: {}'.format(
            working_dir, out, err
        ))


def mount(cli, rev, working_dir, obj_store_path=None):
    return cli.mount_path(path=None,
                          changeset=rev,
                          mount_point=working_dir,
                          object_store_path=obj_store_path,
                          allow_root=True,
                          fetch_all=False)


def get_branch_info(cli, working_dir):
    cmd = [cli.binary_path, 'info', '--json']
    json_data_str = sp.check_output(cmd, cwd=working_dir)
    log.info('arc info: %s', json_data_str)
    json_data = json.loads(json_data_str)
    return json_data.get('branch'), json_data.get('detached')


def get_commit(cli, branch_or_commit_or_revision):
    if branch_or_commit_or_revision.isdigit():
        branch_or_commit_or_revision = 'r' + branch_or_commit_or_revision

    with cli.init_bare() as mp:
        json_log = cli.log(mp,
                           path=ARC_SDC_REL_PATH,
                           start_commit=branch_or_commit_or_revision,
                           max_count=1,
                           as_dict=True)
        if len(json_log) > 0:
            return json_log[0].get('commit')


def get_commits(cli, branch_or_commit_or_revision, count=100):
    if branch_or_commit_or_revision.isdigit():
        branch_or_commit_or_revision = 'r' + branch_or_commit_or_revision

    with cli.init_bare() as mp:
        return cli.log(
            mp,
            path=ARC_SDC_REL_PATH,
            start_commit=branch_or_commit_or_revision,
            max_count=count,
            as_dict=True
        )
