import os
import logging
import boto3
import botocore

from sandbox import common
from sandbox import sdk2
from six.moves.urllib.parse import urlparse, urlunparse
from sandbox.sdk2.helpers import subprocess as sp

log = logging.getLogger(__name__)


def initialize_aws_credentials(key_id, secret_key):
    log.info('Initialize aws credentials')

    credentials_content = [
        '[default]',
        'aws_secret_access_key={}'.format(secret_key),
        'aws_access_key_id={}'.format(key_id),
        '',
    ]

    aws_token_dir = os.path.join(os.path.expanduser('~'), '.aws')
    if not os.path.exists(aws_token_dir):
        os.mkdir(aws_token_dir)
    with open(os.path.join(aws_token_dir, 'credentials'), 'w') as f:
        f.write('\n'.join(credentials_content))

    log.info('Aws done')


def download_ci_and_binaries_from_s3(task, ci_url, binaries_url, secret_access_key, access_key_id):
    log.info('download_ci_and_binaries_from_s3: ci: {} binaries: {}'.format(
        ci_url, binaries_url))

    download_from_s3(task, ci_url, secret_access_key, access_key_id)
    download_from_s3(task, binaries_url, secret_access_key, access_key_id)


def _get_correct_url(url):
    url_parsed = urlparse(url)
    netloc = url_parsed.netloc
    if not url_parsed.scheme or not netloc:
        raise ValueError('{} does not look like url'.format(url))

    path = url_parsed.path

    if url_parsed.netloc.startswith(('s3', 's3-private')):
        path = path.lstrip('/').split('/')
        bucket = path[0]
        key = '/'.join(path[1:])

        return urlunparse((url_parsed.scheme, '{}.{}'.format(bucket, netloc), key, '', '', ''))

    return url


def s3_components_from_url(url):
    """
    Split url in 3 s3 components: endpoint ulr, bucket name, key
    Args:
        url (str):
            Url which we need to split
    Return:
        tuple: (endpoint_url, bucket, key)
    """
    url_parsed = urlparse(url)
    bucket = url_parsed.netloc.split('.')[0]

    # strip first part before '.', because it is bucket name
    endpoint_url = '{}://{}'.format(url_parsed.scheme,
                                    url_parsed.netloc.replace('{}.'.format(bucket), '', 1))
    key = url_parsed.path.lstrip('/')
    return endpoint_url, bucket, key


def download_from_s3(task, mds_url, secret_access_key, access_key_id):
    log.info('download_from_s3: {}'.format(mds_url))
    endpoint_url, bucket, key = s3_components_from_url(_get_correct_url(mds_url))
    b3_client = boto3.session.Session().client('s3', endpoint_url=endpoint_url,
                                               config=botocore.client.Config(
                                                   s3={'addressing_style': 'virtual'},
                                                   retries={'max_attempts': 5}),
                                               aws_access_key_id=access_key_id,
                                               aws_secret_access_key=secret_access_key)

    bin_name = os.path.basename(key)
    try:
        b3_client.download_file(bucket, key, bin_name)
        call(task, ['tar', '-xzf', bin_name])
    except boto3.exceptions.Boto3Error as e:
        log.exception('Some problems, while downloading resource from s3 %s', repr(e))
        raise common.errors.TaskFailure('Cannot download resource')


def call(task, cmd):
    with sdk2.helpers.ProcessLog(task, logger=log) as pl:
        sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT)
