from __future__ import absolute_import

import json
import sys

FIELD_ERROR_CODE = 'error_code'
FIELD_ERROR_MESSAGE = 'error_message'
FIELD_ERROR_DETAILS = 'error_details'
FIELD_LOG = 'log'
FIELD_RESULT = 'result'
FIELD_VERSION = 'version'

ERROR_CODE_API_NOT_SUPPORTED = 'NOT_SUPPORTED'
ERROR_CODE_EXCEPTION = 'EXCEPTION'
ERROR_CODE_USAGE = 'USAGE'


class ApiCallResult(object):
    """This class is DTO for api call result.
    If api call is success then 'result' and 'version' fields are present(with optional 'log' field)
    Otherwise 'error_code' and 'error_message' and 'error_details'
    """

    def __init__(self, result, version, captured_log, error_code, error_message, error_details):
        self._result = result
        self._captured_log = captured_log
        self._version = version
        self._error_code = error_code
        self._error_message = error_message
        self._error_details = error_details or ''

    def __repr__(self):
        return str(self.to_dict())

    @classmethod
    def success(cls, result, version, captured_log=None):
        return cls(result, version, captured_log, None, None, None)

    @classmethod
    def error(cls, message, version=None, code=ERROR_CODE_EXCEPTION, details=''):
        return cls(None, version, None, code, message, details)

    @classmethod
    def from_dict(cls, d, with_log=None):
        result = d.get(FIELD_RESULT)
        captured_log = with_log or d.get(FIELD_LOG)
        version = d.get(FIELD_VERSION)
        error_code = d.get(FIELD_ERROR_CODE)
        error_message = d.get(FIELD_ERROR_MESSAGE)
        error_details = d.get(FIELD_ERROR_DETAILS)
        return ApiCallResult(result, version, captured_log,
                             error_code, error_message, error_details)

    def raise_on_error(self):
        if self.is_error():
            raise BinariesApiError(self)

    def to_dict(self):
        d = {}

        if self._version:
            d[FIELD_VERSION] = self._version

        if self._captured_log:
            d[FIELD_LOG] = self._captured_log

        if self.is_success():
            d[FIELD_RESULT] = self._result
        else:
            d.update({
                FIELD_ERROR_CODE: self._error_code,
                FIELD_ERROR_MESSAGE: self._error_message,
                FIELD_ERROR_DETAILS: self._error_details,
            })

        return d

    def log(self):
        return self._captured_log

    def result(self):
        return self._result

    def version(self):
        return self._version

    def error_code(self):
        return self._error_code

    def is_success(self):
        return not self.is_error()

    def is_error(self):
        return bool(self._error_code)

    def to_stdout(self):
        json.dump(self.to_dict(), sys.stdout)


class BinariesApiError(Exception):
    """Raised if api call result is failed(or finished with error_code)"""

    def __init__(self, api_call_result):
        """
        :type api_call_result: ApiCallResult
        """
        self.api_call_result = api_call_result
        self.message = str(self.api_call_result)
        super(BinariesApiError, self).__init__(self.message)
