from __future__ import absolute_import

import json
import logging
import os
import subprocess
import traceback


from .api_call_result import ApiCallResult
from .api_call_result import ERROR_CODE_API_NOT_SUPPORTED


def execute(root_dir, api_name, cli_args=None, raise_on_error=True):
    # Ensure api runner exists
    api_runner = os.path.join(root_dir, 'ci/api/Run')
    if not os.path.exists(api_runner):
        r = ApiCallResult.error('Binaries API not supported',
                                details='Api runner is not found: {!r}'.format(api_runner),
                                code=ERROR_CODE_API_NOT_SUPPORTED)
        return raise_or_result(r, raise_on_error)

    # Prepare api runner arguments
    call_args = [api_runner, api_name]
    if cli_args:
        call_args.extend(cli_args)

    # Execute api runner
    stdout = ''
    stderr = ''
    try:
        logging.info('Calling binaries API: {}'.format(' '.join(call_args)))
        proc = subprocess.Popen(call_args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        stdout, stderr = proc.communicate()
        ret_code = proc.returncode

        if ret_code == 0:
            # Treat all stdout as JSON, parse api result call, save stderr
            r = json.loads(stdout)
            call_result = ApiCallResult.from_dict(r, with_log=stderr)
        else:
            # Abnormal subprocess failure (e.g. uncaught exception in api impl)
            msg = "'{}' => returns {}".format(' '.join(call_args), ret_code)
            details = make_details([('stdout', stdout), ('stderr', stderr)])
            call_result = ApiCallResult.error(msg, details=details)
    except Exception as e:
        # Abnormal exception in caller code(eg JSON parse exception)
        msg = str(e)
        traceback_str = traceback.format_exc()
        details = make_details([('traceback', traceback_str),
                                ('stdout', stdout),
                                ('stderr', stderr)])
        call_result = ApiCallResult.error(msg, details=details)

    return raise_or_result(call_result, raise_on_error)


def raise_or_result(call_result, raise_on_error):
    if raise_on_error:
        call_result.raise_on_error()

    return call_result


def make_details(parts):
    non_empty_parts = []
    for name, val in parts:
        if val:
            non_empty_parts.append('{}:{}'.format(name, val))
    return '\n'.join(non_empty_parts)
