import logging
import requests

from six.moves.urllib.parse import urljoin

from sandbox.projects.common.decorators import retries
from sandbox.projects.sdc.common import session
from sandbox.projects.sdc.common.constants import DEFAULT_BB_SERVER_URL


class BitbucketClient(object):
    def __init__(self, token, url=DEFAULT_BB_SERVER_URL):
        """
        :param token: token to access to bitbucket
        :type token: str
        """
        self.session = session.create_session()
        self.session.auth = ('x-oauth-token', token)
        self.plain_session = self.plain_session(token)
        self.server_url = url
        self.rest_api_url = urljoin(self.server_url, 'rest/api/1.0/projects/SDC/repos/sdc')
        self.branch_utils_url = urljoin(self.server_url, 'rest/branch-utils/latest/projects/SDC/repos/sdc')

    def plain_session(self, token):
        session = requests.Session()
        session.auth = ('x-oauth-token', token)
        return session

    def get_commit_from_branch_or_commit(self, branch_or_commit):
        """
        Returns last commit on branch if branch was given or full commit hash if full
        or short commit hash was given.

        :param branch_or_commit: string containing branch or full commit hash or short commit hash
        :type branch_or_commit: str

        :rtype str
        """
        resp = self.session.get('{}/commits'.format(self.rest_api_url),
                                params={'until': branch_or_commit, 'limit': 1})
        resp.raise_for_status()
        return resp.json()['values'][0]['id']

    def get_raw_file_by_revision(self, commit, path_to_file, result_filename):
        resp = self.session.get('{bb_url}/raw/{filename}?at={revision}'.format(
            bb_url=self.rest_api_url,
            filename=path_to_file,
            revision=commit))
        resp.raise_for_status()
        with open(result_filename, 'wb') as fd:
            fd.write(resp.content)
            fd.close()
        return result_filename

    @retries(max_tries=5, delay=10)
    def get_branches_for_commit(self, commit, limit=10):
        logging.info('Trying to get branches for commit {}'.format(commit))
        try:
            # we dont want to retry this request, since we expect 500 http code for some cases
            resp = self.plain_session.get('{bb_url}/branches/info/{commit}?limit={limit}'.format(
                bb_url=self.branch_utils_url, commit=commit, limit=limit
            ))
            resp.raise_for_status()
            return map(lambda b: b['id'], resp.json()['values'])
        except BaseException as e:
            logging.exception(e)
            logging.info(resp.json())
            if 'process timed out' in resp.json()['errors'][0]['message']:
                return []
            raise e

    @retries(max_tries=5, delay=10)
    def get_commits_for_branch(self, branch, start=0, limit=100):
        logging.info('Trying to get commits for branch {}'.format(branch))
        try:
            resp = self.plain_session.get('{bb_url}/commits?until={until}&start={start}&limit={limit}'.format(
                bb_url=self.rest_api_url, until=branch, start=start, limit=limit
            ))
            resp.raise_for_status()
            return resp.json()['values']
        except BaseException as e:
            logging.exception(e)
            raise

    def check_commit_exists_in_branch(self, commit, branch):
        branches = self.get_branches_for_commit(commit)
        logging.info('Commit {} exists in branches {}'.format(commit, branches))
        return branch in branches
