import json
import logging

from sandbox.projects.sdc.common.utils import run_command

log = logging.getLogger(__name__)

DOCKER_CONFIG_FILE = '/etc/docker/daemon.json'


def configure_docker(task):
    with open(DOCKER_CONFIG_FILE, 'r') as f:
        docker_config = json.load(f)

    docker_path = str(task.path('docker'))
    log.info('Using task working directory for docker at %s', docker_path)
    docker_config['data-root'] = docker_path
    with open(DOCKER_CONFIG_FILE, 'w') as f:
        json.dump(docker_config, f, indent=4)

    try:
        run_command(task, ['sudo', 'service', 'docker', 'restart'], retries=3, timeout=60)
    except Exception:
        logging.error('restart docker failed. Show status and journal')
        run_command(task, ['systemctl', 'status', 'docker.service'], ignore_bad_exit_code=True)
        run_command(task, ['journalctl', '-xe'], ignore_bad_exit_code=True)
        raise

    run_command(task, ['docker', 'info'])
    run_command(task, ['docker', 'image', 'list'])


def login_to_docker(task, user, token):
    log.info('Login to docker as %s', str(user))
    cmd = [
        'docker', 'login', '-u', str(user),
        '-p', token,
        'registry.yandex.net'
    ]
    run_command(task, cmd, retries=5, retry_delay_sec=60)
    log.info('Logged in')


def load_image(task, image_path):
    log.info('Loading image %s', image_path)
    cmd = ['docker', 'load', '--input', image_path]
    run_command(task, cmd)
    log.info('Image loaded')
