import os
import logging
import sys
import shutil

from sandbox import sdk2
from sandbox.projects.common.decorators import retries
from sandbox.projects.sdc.common.utils import normalize_branch
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs import git

log = logging.getLogger(__name__)


@retries(max_tries=3, delay=10)
def git_cached_checkout(task, repo_url, target_dir, ssh_key, branch, commit):
    with sdk2.ssh.Key(task, private_part=ssh_key):
        # remove everything from target_dir for retrying
        if os.path.exists(target_dir):
            shutil.rmtree(target_dir)
        log.info('Cached clone to %s', str(target_dir))
        repo = git.Git(repo_url, filter_branches=False)
        repo.clone(target_dir, branch, commit)
        normalized_branch = normalize_branch(branch)
        if normalized_branch and not normalized_branch.startswith('pull-requests'):
            # sandbox tries to checkout on ref instead of branch
            call(task, ['git', 'checkout', '-f', normalized_branch], cwd=target_dir)
        call(task, ['git', 'status'], cwd=target_dir)
        call(task, ['git', 'log', '-n', '1'], cwd=target_dir)
        logging.info('Cached git clone finished')
        # add our repository to pythonpath
        sys.path.insert(0, target_dir)
        return repo.cache_repo_dir


def initialize_git():
    git_config = [
        '[user]',
        'name=sandbox',
        'email=sandbox@yandex-team.ru'
    ]
    git_config_path = os.path.join(os.path.expanduser('~'), '.gitconfig')
    with open(git_config_path, 'w') as f:
        f.write('\n\t'.join(git_config))


def get_last_midnight_dev_tag(task, working_dir, ssh_key):
    with sdk2.ssh.Key(task, private_part=ssh_key):
        call(task, ['git', 'fetch', '--tags', 'origin', '+refs/tags/*:refs/tags/*'], cwd=working_dir)
        midnight_dev_tags = sp.check_output(
            ['git', 'tag', '-l', 'midnight-dev-*.*.*', '--sort', '-v:refname'], cwd=working_dir
        )
        return midnight_dev_tags.splitlines()[0]


def get_branch_name(task, working_dir, ssh_key, ref):
    with sdk2.ssh.Key(task, private_part=ssh_key):
        branch_name = sp.check_output(
            ['git', 'name-rev', '--name-only', ref], cwd=working_dir
        )
        return branch_name.splitlines()[0].replace('/merge-pin', '')


def call(task, cmd, cwd=os.getcwd()):
    with sdk2.helpers.ProcessLog(task, logger=log) as pl:
        sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT, cwd=cwd)
