import requests.adapters
from requests.packages.urllib3.util.retry import Retry


def create_session_with_retry_config(
        retry_status_forcelist=None, retry_method_whitelist=None, retries=5, backoff_factor=20, backoff_max=300
):
    class CustomBackoffRetry(Retry):
        def __init__(self, backoff_max=300, **kwargs):
            super(CustomBackoffRetry, self).__init__(**kwargs)
            self.BACKOFF_MAX = backoff_max

    session = requests.Session()
    # backoff_factor=20, backoff_max=300
    # total 5 retries: after 0sec, 40sec, 80sec, 160sec, 300sec
    # See backoff_factor https://urllib3.readthedocs.io/en/latest/reference/urllib3.util.html
    session.mount('https://', requests.adapters.HTTPAdapter(
        max_retries=CustomBackoffRetry(backoff_max=backoff_max, total=retries, backoff_factor=backoff_factor,
                                       status_forcelist=retry_status_forcelist,
                                       method_whitelist=retry_method_whitelist or Retry.DEFAULT_METHOD_WHITELIST)))
    return session


def create_session(retries=5, backoff_factor=20, backoff_max=300):
    """
    Create default session with retries on http 500, 502, 503, 504 codes
    """
    return create_session_with_retry_config(
        retry_status_forcelist=frozenset([500, 502, 503, 504]), retries=retries,
        backoff_factor=backoff_factor, backoff_max=backoff_max
    )


def create_session_with_404_5xx_retries():
    """
    Create session with retries on http 404, 500, 502, 503, 504 codes
    """
    methods_whitelist = list(Retry.DEFAULT_METHOD_WHITELIST)
    methods_whitelist.append('POST')
    return create_session_with_retry_config(
        retry_status_forcelist=frozenset([500, 502, 503, 504, 404]),
        retry_method_whitelist=frozenset(methods_whitelist)
    )
