from xml.etree import ElementTree as et
import logging
import os
import subprocess

import requests

from sandbox.projects.sdc.common import session


class OAuthWithWget(requests.auth.AuthBase):

    def __init__(self, token):
        self.value = 'OAuth ' + token

    def __call__(self, r):
        r.headers['Authorization'] = self.value
        return r

    @property
    def wgetAuth(self):
        return ['--header', 'Authorization: {}'.format(self.value)]


class TeamcityClient(object):
    TEAMCITY_URL = 'https://teamcity.yandex-team.ru'
    TEAMCITY_REST_URL = '{}/app/rest'.format(TEAMCITY_URL)

    def __init__(self, session_auth):
        """
        :param username: username to access to teamcity
        :type username: str
        :param password: password to access to teamcity
        :type password: str
        :param oauth_token
        :type oauth_token: str
        """
        self.session = session.create_session()
        self.session.auth = session_auth

        self.session_with_404_retry = session.create_session_with_404_5xx_retries()
        self.session_with_404_retry.auth = session_auth

    @classmethod
    def with_oauth(cls, oauth_token):
        return cls(OAuthWithWget(oauth_token))

    @classmethod
    def artifact_url(cls, build_id, artifact_path):
        """
        Constructs artifacts url from build id and path to artifact.

        :param build_id: teamcity build id
        :type build_id: str or int
        :param artifact_path: path to teamcity artifact
        :type artifact_path: str

        :rtype: str
        """
        return '{}/builds/id:{}/artifacts/content/{}'.format(cls.TEAMCITY_REST_URL, build_id, artifact_path)

    @classmethod
    def build_url(cls, build_id):
        """
        Returns teamcity build url constructed from build id.

        :param build_id: teamcity build id
        :type build_id: str or int

        :rtype str
        """
        return '{}/viewLog.html?buildId={}'.format(cls.TEAMCITY_URL, build_id)

    def builds(self, locator):
        """
        Returns builds suitable for given locator.

        :param locator: teamcity build locator
        :type locator: str

        :rtype ElementTree
        """
        resp = self.session.get('{}/builds'.format(self.TEAMCITY_REST_URL), params={'locator': locator})
        resp.raise_for_status()
        return et.fromstring(resp.text.encode('utf-8'))

    def build_details(self, build_id):
        """
        Downloads build artifact to file with the same name in current directory.

        :param build_id: teamcity build id
        :type build_id: str or int

        :rtype ElementTree
        """
        resp = self.session.get('{}/builds/id:{}'.format(self.TEAMCITY_REST_URL, build_id))
        resp.raise_for_status()
        tree = et.fromstring(resp.text.encode('utf-8'))
        return tree

    def build_resulting_props(self, build_id):
        """
        Downloads build artifact to file with the same name in current directory.

        :param build_id: teamcity build id
        :type build_id: str or int

        :rtype ElementTree
        """
        resp = self.session.get('{}/builds/id:{}/resulting-properties'.format(self.TEAMCITY_REST_URL, build_id))
        resp.raise_for_status()
        tree = et.fromstring(resp.text.encode('utf-8'))
        return tree

    def check_access(self):
        """
        Checks whether teamcity server is available and credentials are accepted.
        """
        resp = self.session.head(self.TEAMCITY_REST_URL)
        resp.close()
        return resp.status_code

    def download_artifact(self, build_id, artifact_path):
        """
        Downloads build artifact to file with the same name in current directory.

        :param build_id: teamcity build id
        :type build_id: str or int
        :param artifact_path: path to teamcity artifact
        :type artifact_path: str
        """
        # wget it used here to get continuous downloading for free
        command = ['wget', '--quiet', '-t', '10', '--retry-connrefused', '-c', '-O',
                   os.path.basename(artifact_path)]
        command.extend(self.session.auth.wgetAuth)
        command.append(self.artifact_url(build_id, artifact_path))
        subprocess.check_call(command)

    def trigger_build(self, build_conf, branch=None, last_changes=None, params=None):
        """
        Triggers build on teamcity.

        :param build_conf: teamcity build configuration id
        :type build_conf: str
        :param branch: branch to trigger build
        :type branch: str or NoneType
        :param last_changes: changes to trigger build
        :type last_changes: str or NoneType
        :param params: parameters for triggering build
        :type params: dict or NoneType

        :rtype ElementTree
        """
        build_el = et.Element('build')
        et.SubElement(build_el, 'buildType').set('id', build_conf)
        if branch:
            build_el.set('branchName', branch)
        if last_changes:
            last_change_el = et.SubElement(build_el, 'lastChanges')
            et.SubElement(last_change_el, 'change').set(
                'locator', 'version:{},buildType:(id:{})'.format(last_changes, build_conf))
        if params:
            properties_el = et.SubElement(build_el, 'properties')
            for param in params:
                property_el = et.SubElement(properties_el, 'property')
                property_el.set('name', param)
                property_el.set('value', str(params[param]))

        # We using here session_with_404_retry because teamcity.yandex-team.ru/app/rest/buildQueue
        # should not answer 404. 404 most likely that commit is not checkouted by TC yet, so retry.
        # https://st.yandex-team.ru/SDC-25530
        resp = self.session_with_404_retry.post(
            'https://teamcity.yandex-team.ru/app/rest/buildQueue',
            data=et.tostring(build_el),
            headers={
                'Content-Type': 'application/xml',
                'Origin': 'http://teamcity.yandex-team.ru:443',
            })
        resp.raise_for_status()
        return et.fromstring(resp.text.encode('utf-8'))

    def cancel_build(self, build_id, comment=''):
        logging.info('Try to stop build {}'.format(build_id))
        stopped = self.cancel_queued_build(build_id, comment=comment)
        if not stopped:
            stopped = self.cancel_running_build(build_id, comment=comment)
        if stopped:
            logging.info('Build {} was stopped'.format(build_id))
        else:
            logging.info('Failed to cancel build {}'.format(build_id))
        return stopped

    def cancel_queued_build(self, build_id, comment='', re_add=False):
        return self._cancel_build('buildQueue', build_id, comment, re_add)

    def cancel_running_build(self, build_id, comment='', re_add=False):
        return self._cancel_build('builds', build_id, comment, re_add)

    def _cancel_build(self, tc_url, build_id, comment='', re_add=False):
        data = et.Element('buildCancelRequest')
        data.set('comment', comment)
        data.set('readdIntoQueue', str(re_add).lower())
        try:
            r = self.session.post(
                '{baseUrl}/{url}/id:{id}/'.format(baseUrl=self.TEAMCITY_REST_URL, url=tc_url, id=build_id),
                data=et.tostring(data),
                headers={
                    'Content-Type': 'application/xml',
                    'Origin': 'http://teamcity.yandex-team.ru:443',
                })
            r.raise_for_status()
        except requests.exceptions.HTTPError as e:
            print('Error: {}'.format(e))
            return False
        return True
