import logging
import os
import shlex
import socket
import sys
import time
from datetime import datetime
from subprocess import Popen, PIPE

from sandbox.sandboxsdk.environments import VirtualEnvironment

from sandbox import sdk2
from sandbox.sdk2.helpers import gdb, subprocess


def configure_network():
    with open('/etc/hosts', 'r') as hosts_fd:
        hosts_content = hosts_fd.read()

    hostname = socket.gethostname()

    line_to_add = '127.0.0.1\t{}'.format(hostname)
    if line_to_add not in hosts_content:
        p = Popen(['sudo', 'tee', '-a', '/etc/hosts'], stdin=PIPE)
        p.stdin.write(line_to_add.encode())
        p.stdin.close()
        p.wait()


def configure_env(venv=None):
    # in our build script we use shebang #!/usr/bin/env python, which will use
    # the first python from PATH. So, we can use that to change python to python
    # from SANDBOX in order to have all SANDBOX libraries
    python_executable = venv.executable if venv else sys.executable
    os.environ['PATH'] = '{}{}{}'.format(
        os.path.dirname(python_executable), os.pathsep, os.getenv('PATH')
    )


def normalize_branch(branch):
    """Teamcity plugin adds refs/ before branch name, so we need to remove that"""
    branch = str(branch) if branch else ''
    if branch.startswith('refs/heads/'):
        branch = branch.replace('refs/heads/', '')
    elif branch.startswith('refs/pull-requests/'):
        branch = branch.replace('refs/', '').replace('/merge-pin', '')
    return branch


def create_venv():
    return VirtualEnvironment(
        python_exe='/usr/bin/python', use_system=True,
        venv_args=('-m', 'virtualenv', '--no-setuptools', '--no-pip', '--no-wheel')
    )


class RunCommandException(Exception):
    def __init__(self, cmd):
        super(RunCommandException, self).__init__('Failed to execute command {}'.format(cmd))


def run_command(
    task,
    cmd,
    env=None,
    cwd=None,
    logger=None,
    print_logs=False,
    timeout=None,
    retries=0,
    retry_delay_sec=0,
    shell=False,
    ignore_bad_exit_code=False,
):
    if not isinstance(cmd, list) and not shell:
        cmd = shlex.split(cmd)
    if not logger:
        cmd_parts = cmd if not shell else cmd.split()
        logger = cmd_parts[0]
        if logger == 'sudo' and len(cmd_parts) > 1:
            logger = cmd_parts[1]

    for attempt in range(retries + 1):
        with sdk2.helpers.ProcessLog(task, logger=logger) as pl:
            if print_logs:
                task.set_info(
                    gdb.get_html_view_for_logs_file(
                        'log',
                        pl.stdout.path.relative_to(task.log_path()),
                        task.log_resource
                    ),
                    do_escape=False,
                )
            start = time.time()
            try:
                subprocess.check_call(
                    cmd, env=env, cwd=cwd,
                    stdout=pl.stdout, stderr=subprocess.STDOUT,
                    timeout=timeout, shell=shell,
                )
            except Exception as e:
                logging.warning('Failed to execute command %s on %s try: %s', cmd, attempt + 1, e)
                if retry_delay_sec > 0:
                    time.sleep(retry_delay_sec)
            else:
                break
            finally:
                total_time = int(time.time() - start)
                logging.info(
                    'Execution time of the command %s: %s ',
                    cmd,
                    time.strftime('%X', datetime.utcfromtimestamp(total_time).timetuple()),
                )
    else:
        # print logs for error debugging
        if not print_logs:
            task.set_info(
                gdb.get_html_view_for_logs_file(
                    'log',
                    pl.stdout.path.relative_to(task.log_path()),
                    task.log_resource,
                ),
                do_escape=False,
            )
        if ignore_bad_exit_code:
            return
        raise RunCommandException(cmd)
