import os
import logging

log = logging.getLogger(__name__)

YT_VERSION = '0.9.23'


def initialize_yt_credentials(yt_token):
    log.info('Initialize YT token')
    yt_token_dir = os.path.join(os.path.expanduser('~'), '.yt')
    if not os.path.exists(yt_token_dir):
        os.mkdir(yt_token_dir)
    with open(os.path.join(yt_token_dir, 'token'), 'w') as f:
        f.write(yt_token)
    log.info('YT done')


def get_yt_client(cluster, token):
    os.environ['YT_LOG_LEVEL'] = 'DEBUG'
    import yt.wrapper as yt
    return yt.YtClient(cluster, token=token)


class YtCluster(object):
    def __init__(self, cluster_name, token):
        self.name = cluster_name
        self.client = get_yt_client(cluster_name, token)


def create_clusters(cluster_names, token):
    clusters = []
    for cluster_name in cluster_names:
        clusters.append(YtCluster(cluster_name, token))
    return clusters


def get_cluster_unavailable_errors():
    from yt.packages.requests import ConnectionError, ReadTimeout
    from yt.wrapper import errors
    return (
        ConnectionError, ReadTimeout,
        errors.YtProxyUnavailable,
        errors.YtRpcUnavailable,
        errors.YtRequestTimedOut,
    )


def check_yt_path_existence(clusters, yt_path):
    cluster_unavailable_errors = get_cluster_unavailable_errors()
    for yt_cluster in clusters:
        try:
            if not yt_cluster.client.exists(yt_path):
                return False
        except cluster_unavailable_errors as e:
            # ignore connection error due to cluster unavailability,
            # assuming, that only one cluster can be unavailable at the time
            logging.info('Cluster {} is unavailable'.format(yt_cluster.name))
            logging.exception(e)
    return True
