import json
import logging
import os
import time

import six

from sandbox import common
from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common.teamcity import TeamcityArtifactsContext
from sandbox.projects.common.task_env import TinyRequirements
from sandbox.projects.sdc.common import utils
from sandbox.projects.sdc.common import arc
from sandbox.projects.sdc.common import constants
from sandbox.projects.sdc.common.aws import initialize_aws_credentials
from sandbox.projects.sdc.common.bitbucket import BitbucketClient
from sandbox.projects.sdc.common.constants import (
    DEFAULT_BB_SERVER_URL,
    DEFAULT_REPO_URL,
    GITHUB_REPO_URL,
    ROBOT_SDC_CI_SECRET_ID,
    TEAMCITY_BUILD_URL_PATTERN,
)
from sandbox.projects.sdc.common.docker import login_to_docker, configure_docker
from sandbox.projects.sdc.common.git import git_cached_checkout, initialize_git
from sandbox.projects.sdc.common.yt_helpers import initialize_yt_credentials
from sandbox.projects.sdc.resource_types import SdcLxcContainer
from sandbox.sdk2 import yav
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.common.types.misc as ctm

log = logging.getLogger(__name__)


class SdcArtifactFileResource(sdk2.Resource):
    DEFAULT_TTL = 7

    ttl = DEFAULT_TTL
    file_name = sdk2.Attributes.String('File name', required=True)
    sync_upload_to_mds = True


class BaseSdcTask(sdk2.Task):
    CI_REPORT_HTML = 'ci_report.html'

    class Requirements(TinyRequirements):
        # Root required for configure docker
        # BUT privileged=True is not working here from the box
        # See also:
        # https://st.yandex-team.ru/RMDEV-2279#6088381710982736513ff2ef
        # https://st.yandex-team.ru/ARC-2601
        # + 0.sh will be called from root, but docker is failed in that case:
        # see sdc/infra/docker/__init__.py [ self.permission_check() ]

        # https://st.yandex-team.ru/DEVTOOLSSUPPORT-16872
        cores = 17  # 1 is ok, but > 16 is required to prevent docker failures on multislot
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Parameters):
        dump_disk_usage = False

        # VCS Section
        vcs_type = sdk2.parameters.String('VCS type', default=constants.DEFAULT_VCS_TYPE)
        arc_mode = sdk2.parameters.String('Arc mode', default=constants.DEFAULT_ARC_MODE)
        branch = sdk2.parameters.String(
            'Teamcity build branch (with refs/heads/ prefix)',
            default='refs/heads/dev',
            hint=True,
        )
        # Can't use hint here, due: https://st.yandex-team.ru/SDC-96446#6242cb1ea3337b22d4864356
        # Hint from sdc_vcs_commit or build_commit(if present) will be used
        commit = sdk2.parameters.String('Teamcity build commit')
        build_branch = sdk2.parameters.String('Teamcity build branch (without vcs)', hint=True)
        build_commit = sdk2.parameters.String('Teamcity build commit (without vcs)', hint=True)
        repo_url = sdk2.parameters.String('SDC repo url', default=DEFAULT_REPO_URL)
        github_repo_url = sdk2.parameters.String('SDC github repo url', default=GITHUB_REPO_URL)
        bb_server_url = sdk2.parameters.String(
            'Bitbucket server base url',
            default=DEFAULT_BB_SERVER_URL,
        )
        # Buildscript info
        git_steps_path = sdk2.parameters.String('Path to 0.sh script to Run')
        vcs_steps_path = sdk2.parameters.String('Path to 0.sh script to Run (vcs steps)')

        # Common SDC build_scripts parameters
        build_type = sdk2.parameters.String('Build type', default='RELEASE')
        build_type_id = sdk2.parameters.String('Teamcity build type id', hint=True)
        build_type_name = sdk2.parameters.String('Teamcity build type name')
        parent_build_id = sdk2.parameters.String('Parent build id')
        parent_build_type_id = sdk2.parameters.String('Teamcity parent build type id', default='')
        parent_build_url = sdk2.parameters.String('Teamcity parent build url', default='')
        teamcity_version = sdk2.parameters.String(
            'Teamcity version (for emulating teamcity agent)',
            default='SANDBOX', required=True
        )

        # sdc_mon clickhouse parameters
        ch_sdc_mon_user = sdk2.parameters.String(
            'sdc_mon clickhouse user (should correspond to secret ch_sdc_mon_password)',
            required=False, default='robot-sdc-ci',
        )

        # Legacy SDC build_script parameters (should be replaced by env_context.*)
        branch_re = sdk2.parameters.String('Branch regexp')
        build_args = sdk2.parameters.String('Build arguments')
        ctest_disabled = sdk2.parameters.String('Disable ctest running')
        force_publish_logs = sdk2.parameters.String('Force publish logs')
        commit_hash = sdk2.parameters.String('Commit hash')
        test = sdk2.parameters.String('Test')
        package_version = sdk2.parameters.String('Package version')

        # Teamcity integration
        teamcity_build_id = sdk2.parameters.String('Teamcity build id')

        # Sandbox ENV settings
        username = sdk2.parameters.String('Username', default='sandbox')
        container = sdk2.parameters.Container('LXC container with Docker',
                                              resource_type=SdcLxcContainer)
        env_variables = sdk2.parameters.Dict('Env variables to set')

        # Output settings
        artifacts_as_resources = sdk2.parameters.List(
            'Teamcity artifact paths to publish as sandbox separate resource',
            default=[],
        )

        artifacts_as_json_output = sdk2.parameters.List(
            'Teamcity artifact paths to publish as json output parameter',
            default=[],
        )

        # If force_resources_ttl present - overwrite ALL resources TTL with given value
        force_resources_ttl = sdk2.parameters.Integer('Custom resources ttl')
        # Teamcity log TTL
        buildscript_log_ttl = sdk2.parameters.Integer('Buildscript log ttl', default=14)
        # Teamcity artifacts ttl (see: publish_artifacts_from_sandbox.py)
        buildscript_artifacts_ttl = sdk2.parameters.Integer('Buildscript artifacts ttl', default=7)
        # Direct link artifacts ttl
        artifacts_as_resources_ttl = sdk2.parameters.Integer(
            'Artifacts as resources ttl',
            default=SdcArtifactFileResource.DEFAULT_TTL,
        )

        with sdk2.parameters.Output(reset_on_restart=True):
            runtime_parameters = sdk2.parameters.Dict("Collected runtime parameters")
            runtime_statistics = sdk2.parameters.Dict("Collected runtime statistics")
            runtime_build_problems = sdk2.parameters.JSON("Collected runtime build problems")
            runtime_json_output = sdk2.parameters.JSON("Collected runtime JSON output")
            sdc_vcs_commit = sdk2.parameters.String('Actual sdg/sdc vcs commit', hint=True)

    @property
    def artifacts_with_direct_link(self):
        r = []
        r.extend(self.Parameters.artifacts_as_resources)
        r.append('**/{}'.format(self.CI_REPORT_HTML))
        return r

    def on_finish(self, prev_status, status):
        custom_ttl = self.Parameters.force_resources_ttl
        if not custom_ttl:
            return

        # Find all resources for given task and overwrite TTL
        for resource in sdk2.Resource.find(task=self).limit(0):  # Limit 0 means unlimited
            # TODO: We may have additional filtering here (eg by known resource types)
            resource.ttl = custom_ttl

    def on_save(self):
        if not self.Parameters.container:
            self.Parameters.container = SdcLxcContainer.find(
                attrs={"released": "stable"}).first().id

    def on_prepare(self):
        self.logger = logging.getLogger(self.__class__.__name__)

        secrets = yav.Yav(
            robot_sdc_ci=yav.Secret(ROBOT_SDC_CI_SECRET_ID),
            robot_sdc_www=yav.Secret('sec-01e48y5rqvbc7hkys8r28qdj5f'),
            avride_robot_optimus_prime=yav.Secret('sec-01g7rrtq2ee1c56b1e1q009wny')
            # do not forget to delegate new tokens
            # Hide new tokens in log: add them to secret_list()
        )
        self.arc_token = secrets.robot_sdc_ci['token.arc']
        self.aws_key_id = secrets.robot_sdc_ci['aws.key_id']
        self.aws_secret_key = secrets.robot_sdc_ci['aws.secret']
        self.bb_token = secrets.robot_sdc_ci['token.bb']
        self.ch_sdc_mon_password = secrets.robot_sdc_ci['clickhouse.sdc_mon.password']
        self.logbroker_token = secrets.robot_sdc_ci['token.logbroker']
        self.nirvana_token = secrets.robot_sdc_ci['token.nirvana']
        self.sandbox_token = secrets.robot_sdc_ci['token.sandbox']
        self.reactor_token = self.nirvana_token
        self.registry_token = secrets.robot_sdc_ci['token.registry']
        self.ssh_key = secrets.robot_sdc_ci['ssh']
        self.github_ssh_key = secrets.avride_robot_optimus_prime['ssh_private']
        self.st_token = secrets.robot_sdc_ci['token.startrek']
        self.staff_token = secrets.robot_sdc_ci['token.staff']
        self.teamcity_token = secrets.robot_sdc_ci['token.teamcity']
        self.yt_token = secrets.robot_sdc_ci['token.yt']
        self.yql_token = secrets.robot_sdc_ci['token.yql']
        self.awacs_oauth_token = secrets.robot_sdc_www['AWACS_OAUTH_TOKEN']
        self.dctl_yp_token = secrets.robot_sdc_www['DCTL_YP_TOKEN']
        self.emergency_token = secrets.robot_sdc_www['EMERGENCY_OAUTH_TOKEN']
        self.qtools_token = secrets.robot_sdc_www['QTOOLS_OAUTH_TOKEN']
        self.taximeter_token = secrets.robot_sdc_www['TAXIMETER_OAUTH_TOKEN']
        self.www_bb_token = secrets.robot_sdc_www['BITBUCKET_OAUTH_TOKEN']
        self.www_st_token = secrets.robot_sdc_www['TRACKER_OAUTH_TOKEN']
        self.ya_token = secrets.robot_sdc_www['YA_TOKEN']

    def get_yandexsdc_path(self):
        return str(self.path('yandexsdc'))

    @property
    def git_ssh_key(self):
        if self.is_github():
            return self.github_ssh_key
        return self.ssh_key

    @property
    def commit(self):
        return (
            self.Parameters.sdc_vcs_commit
            or self.Parameters.build_commit
            or self.Parameters.commit
        )

    @property
    def branch(self):
        return self.Parameters.build_branch or self.Parameters.branch

    @property
    def build_id(self):
        return self.Parameters.teamcity_build_id or self.id

    @property
    def build_url(self):
        url_pattern = TEAMCITY_BUILD_URL_PATTERN
        if not self.Parameters.teamcity_build_id:
            url_pattern = 'https://sandbox.yandex-team.ru/task/{}'
        return url_pattern.format(self.build_id)

    @property
    def build_type_id(self):
        return self.Parameters.build_type_id or str(self.type)

    @property
    def build_type_name(self):
        return self.Parameters.build_type_name or str(self.type)

    @property
    def parent_build_type_id(self):
        return self.Parameters.parent_build_type_id

    @property
    def parent_build_url(self):
        return self.Parameters.parent_build_url

    def get_buildsdcript_env_variables(self):
        teamcity_integration_env_context = self.Context.sandbox_task_launcher_context

        env_ctx = {}
        # Some variables has default values
        env_ctx.update(self.get_default_env_variables())

        if teamcity_integration_env_context != ctm.NotExists:
            env_ctx.update(teamcity_integration_env_context)

        env_ctx.update(self.Parameters.env_variables)
        env_ctx.update(self.get_non_overridable_env_variables())

        return env_ctx

    def get_default_env_variables(self):
        """
        May be overriden by teamcity env_context.* variables
        :returns: Common SDC buildscript env
        """
        return {
            'T__UPLOAD_PORTO_LAYER': 'false',
            'T__BUILD_TYPE': self.Parameters.build_type,
            'T__BRANCH_RE': self.Parameters.branch_re,
            'T__BUILD_ARGS': self.Parameters.build_args,
            'T__CTEST_DISABLED': self.Parameters.ctest_disabled,
            'T__FORCE_PUBLISH_LOGS': self.Parameters.force_publish_logs,
            'T__PACKAGE_VERSION': self.Parameters.package_version,
            'T__PARENT_BUILD_ID': self.Parameters.parent_build_id,
            'T__TEAMCITY_BUILD_BRANCH': utils.normalize_branch(self.branch),
            'T__COMMIT_HASH': self.Parameters.commit,
            'T__TEST': self.Parameters.test,
            'T__TEAMCITY_BUILD_TYPE_ID': self.build_type_id,
            'T__TEAMCITY_BUILD_TYPE_NAME': self.build_type_name,
            'T__PARENT_BUILD_TYPE_ID': self.parent_build_type_id,
            'T__PARENT_BUILD_URL': self.parent_build_url,
        }

    def get_non_overridable_env_variables(self):
        """
        Can't be overriden by teamcity env_context.* variables
        :returns: Sandbox-specific env variables
        """
        yandexsdc_path = self.get_yandexsdc_path()
        sandbox_client_host = common.config.Registry().this.fqdn

        envs = {
            # Vcs checkout info
            'ARC_MODE': self.Parameters.arc_mode,
            'VCS_TYPE': constants.VCS_ARC if self.is_arc() else constants.VCS_GIT,

            # Sandbox specific envs
            'TEAMCITY_VERSION': self.Parameters.teamcity_version,
            'T__YANDEXSDC_PATH': yandexsdc_path,
            'T__TEAMCITY_AGENT_NAME': sandbox_client_host,
            'REQUESTS_CA_BUNDLE': '/etc/ssl/certs/ca-certificates.crt',

            # Other variables
            'T__CH_SDC_MON_USER': self.Parameters.ch_sdc_mon_user,

            # Secrets - from yav
            'T__BB_TOKEN': self.bb_token,
            'T__CH_SDC_MON_PASS': self.ch_sdc_mon_password,
            'T__KEY_ID': self.aws_key_id,
            'T__KEY_SECRET': self.aws_secret_key,
            'T__SECRET__ARC_TOKEN': self.arc_token,
            'T__SECRET__BB_TOKEN': self.bb_token,
            'T__SECRET__LOGBROKER_OAUTH_TOKEN': self.logbroker_token,
            'T__SECRET__NIRVANA_TOKEN': self.nirvana_token,
            'T__SECRET__ST_TOKEN': self.st_token,
            'T__SECRET__STAFF_TOKEN': self.staff_token,
            'T__SECRET__TEAMCITY_TOKEN': self.teamcity_token,
            'T__SECRET__SANDBOX_TOKEN': self.sandbox_token,
            'T__SECRET__YQL_TOKEN': self.yql_token,
            'T__SECRET__REACTOR_TOKEN': self.reactor_token,
            # If build is used via %syntax% it returns -1, so we use actual values here
            'T__TEAMCITY_BUILD_ID': self.build_id,
            'T__TEAMCITY_BUILD_URL': self.build_url,
            'T__TEAMCITY_BUILD_COMMIT': self.commit,

            # todo: remove after usages of these variables are absent in master-lts,
            #  since for old branches we still have shell scripts with 'set -u' for these variables
            'T__DB_PASS': 'to-be-deleted',
        }

        git_cache_dir = self.Context.git_cache_dir
        if git_cache_dir != ctm.NotExists:
            envs['GIT_CACHE_DIR'] = git_cache_dir

        arc_sdc_cache_dir = self.Context.arc_sdc_cache_dir
        if arc_sdc_cache_dir != ctm.NotExists:
            envs['ARC_SDC_CACHE_DIR'] = arc_sdc_cache_dir

        arc_obj_store_dir = self.Context.arc_object_store
        if arc_obj_store_dir != ctm.NotExists:
            envs['ARC_OBJECT_STORE'] = arc_obj_store_dir

        return envs

    def get_vcs_checkout_dir(self):
        return self.path('vcs')

    def get_working_dir(self):
        vcs_checkout_dir = self.get_vcs_checkout_dir()
        if self.is_arc():
            return vcs_checkout_dir.joinpath('sdg', 'sdc')
        return vcs_checkout_dir

    def get_artifacts_dir(self):
        working_dir = self.get_working_dir()
        return working_dir.joinpath('artifacts')

    def get_build_script_path(self):
        path_parts = [str(self.get_working_dir())]
        path_parts.extend(['ci', 'teamcity', 'Run'])

        script_path = os.path.join(*path_parts)
        return script_path

    def secret_list(self):
        return [
            self.arc_token,
            self.awacs_oauth_token,
            self.aws_key_id,
            self.aws_secret_key,
            self.bb_token,
            self.ch_sdc_mon_password,
            self.dctl_yp_token,
            self.emergency_token,
            self.logbroker_token,
            self.nirvana_token,
            self.qtools_token,
            self.st_token,
            self.staff_token,
            self.taximeter_token,
            self.teamcity_token,
            self.www_bb_token,
            self.www_st_token,
            self.ya_token,
            self.yql_token,
            self.sandbox_token,
            self.reactor_token,
        ]

    def get_buildscript_subprocess_cmd(self, steps_paths=None, env_variables=None):
        if not env_variables:
            env_variables = self.get_buildsdcript_env_variables()

        path_to_build_script = self.get_build_script_path()
        logging.info('build script path: %s', path_to_build_script)

        steps_paths_val = steps_paths or self.Parameters.vcs_steps_path
        steps_paths_params = sum((['-p', p] for p in steps_paths_val.split(',')), [])

        cmd_to_call = [path_to_build_script] + steps_paths_params + ['-E']

        env = []
        for k, v in six.iteritems(env_variables):
            env.append('{}={}'.format(k, v))
        cmd_to_call.extend(env)

        return cmd_to_call

    def run_build_script(self, steps_paths=None, log_name=None):
        working_dir = self.get_working_dir()
        cmd_to_call = self.get_buildscript_subprocess_cmd(steps_paths=steps_paths)

        retcode = 0

        with sdk2.ssh.Key(self, private_part=self.git_ssh_key):
            tac = TeamcityArtifactsContext(
                base_dir=self.get_artifacts_dir(),
                log_name=log_name,
                secret_tokens=self.secret_list(),
                tc_service_messages_ttl=int(self.Parameters.buildscript_log_ttl),
                tc_artifacts_ttl=int(self.Parameters.buildscript_artifacts_ttl),
            )

            with tac:
                try:
                    sp.check_call(
                        cmd_to_call,
                        cwd=str(working_dir),
                        stdout=tac.output,
                        stderr=tac.output,
                    )
                except sp.CalledProcessError as e:
                    retcode = e.returncode

            # Convert to sdk2.Parameter.JSON repr
            build_problems = []
            for message, identity in tac.build_problems:
                build_problems.append({'message': message, 'identity': identity})
            self.Parameters.runtime_build_problems = build_problems

            self.Parameters.runtime_parameters = tac.build_parameters
            self.Parameters.runtime_statistics = tac.build_statistics

            # Emulate teamcity behavior: fail task if build problems present(even retcode is zero)
            if retcode == 0 and build_problems:
                retcode = 100  # see BUILD_PROBLEM_AWARE_CODE from infra.ci.teamcity_helper

        return retcode

    def is_github(self):
        return self.Parameters.vcs_type == constants.VCS_GITHUB

    def is_arc(self):
        return self.Parameters.vcs_type == constants.VCS_ARC

    def is_arc_mount(self):
        return self.Parameters.arc_mode == constants.ARC_MODE_MOUNT

    def vcs_checkout(self):
        if self.is_arc():
            return self.arc_checkout()
        elif self.is_github():
            self.github_checkout()
            return None
        else:
            self.git_checkout()
            return None

    def set_vcs_resulting_parameters(self):
        if not self.is_arc():
            return

        vcs_dir = str(self.get_vcs_checkout_dir())

        top_commit_info, sdc_commit_info = arc.extract_commit_info(self.arc_token, vcs_dir)

        top_commit = top_commit_info['commit']
        sdc_commit = sdc_commit_info['commit']

        self.logger.info('Actual sdc commit is: %r (trunk: %r)', sdc_commit, top_commit)

        self.Parameters.sdc_vcs_commit = sdc_commit

    def arc_checkout(self):
        arc_mode = self.Parameters.arc_mode
        logging.info('arc checkout, mode: %s', arc_mode)
        try:
            rev = self.commit or self.branch

            vcs_dir = str(self.get_vcs_checkout_dir())
            obj_store_dir = str(self.path('object_store'))
            logging.info('working dir: %s, object store dir: %s', vcs_dir, obj_store_dir)
            mp = arc.sdc_checkout(self.arc_token, arc_mode, rev, vcs_dir, obj_store_dir)

            self.Context.arc_object_store = obj_store_dir
            self.Context.arc_sdc_cache_dir = vcs_dir
            return mp
        except Exception:
            logging.exception('Arc checkout exception occurs')
            raise errors.TaskFailure('Arc checkout failed')

    def git_checkout(self):
        try:
            initialize_git()
            bb_client = BitbucketClient(self.bb_token, self.Parameters.bb_server_url)
            branch = self.branch
            if self.commit and utils.normalize_branch(branch) == 'dev':
                # clear branch if we know that commit doesnt belong to dev branch
                if not bb_client.check_commit_exists_in_branch(self.commit, branch):
                    logging.info('Commit {} doesnt belong to {} branch'.format(self.commit, branch))
                    branch = None
            cache_repo_dir = git_cached_checkout(
                self, self.Parameters.repo_url, str(self.get_vcs_checkout_dir()),
                self.ssh_key, branch, self.commit
            )
            self.Context.git_cache_dir = cache_repo_dir
        except Exception:
            logging.exception('Git checkout exception occurs')
            raise errors.TaskFailure('Git checkout failed')

    def github_checkout(self):
        try:
            before_ts = time.time()
            initialize_git()
            branch_to_checkout = self.branch
            if branch_to_checkout.startswith('pull/'):
                branch_to_checkout = branch_to_checkout + '/head'
            cache_repo_dir = git_cached_checkout(
                self,
                self.Parameters.github_repo_url,
                str(self.get_vcs_checkout_dir()),
                self.github_ssh_key,
                branch_to_checkout,
                self.commit
            )
            logging.info('Github checkout takes %s', time.time() - before_ts)
            self.Context.git_cache_dir = cache_repo_dir
        except Exception:
            logging.exception('Git checkout exception occurs')
            raise errors.TaskFailure('GitHub checkout failed')

    def on_build_script_success(self):
        pass

    def on_build_script_failure(self):
        pass

    def on_build_script_finish(self, is_success):
        self.Parameters.runtime_json_output = self.get_runtime_json_output()
        uploaded_resources = self.upload_resources()
        if uploaded_resources:
            self.on_resources_ready(uploaded_resources)

    def on_resources_ready(self, uploaded_resources):
        pass

    def upload_resources(self):
        files_to_separate_artifact = self.get_artifacts_for_separate_resource_upload()
        uploaded_resources = []
        for artifact_path in files_to_separate_artifact:
            file_name = artifact_path.name
            try:
                # create resource & write data
                resource = SdcArtifactFileResource(
                    task=self,
                    description=file_name,
                    path=file_name,
                    file_name=file_name,
                    ttl=int(self.Parameters.artifacts_as_resources_ttl),
                )
                resource.path.write_bytes(artifact_path.read_bytes())
                # Mark resource as ready
                data = sdk2.ResourceData(resource)
                data.ready()

                uploaded_resources.append(resource)
            except Exception:
                self.logger.exception('Got exception for resource upload %r', file_name)

        return uploaded_resources

    def get_runtime_json_output(self):
        files_to_json_output = self.get_artifacts_for_json_output()
        json_output = {}
        for file in files_to_json_output:
            with file.open() as fd:
                json_file_payload = json.load(fd)
                json_output[file.name] = json_file_payload
        return json_output

    def get_artifacts_for_separate_resource_upload(self):
        return self.find_artifacts_match_patterns(self.artifacts_with_direct_link)

    def get_artifacts_for_json_output(self):
        patterns = self.Parameters.artifacts_as_json_output
        return self.find_artifacts_match_patterns(patterns)

    def find_artifacts_match_patterns(self, patterns):
        if not patterns:
            return []

        # Scan artifacts dir, look for files match any of given patterns
        artifacts_dir = self.get_artifacts_dir()
        if not artifacts_dir.exists():
            return []

        found_files = []
        for file in artifacts_dir.iterdir():
            for pattern in patterns:
                if file.match(pattern):
                    found_files.append(file)
                    break  # File match -> no need to continue

        return found_files

    def prepare_docker(self):
        try:
            configure_docker(self)
        except Exception:
            self.logger.exception('Configure docker failed')
            # https://st.yandex-team.ru/SDC-100516
            raise errors.TemporaryError('docker configure failed')
        login_to_docker(self, self.Parameters.username, self.registry_token)

    def on_execute(self):
        with utils.create_venv() as venv:
            self.logger.info('Task {} started'.format(self.__class__.__name__))
            utils.configure_env(venv)
            utils.configure_network()
            self.prepare_docker()
            mount_point = self.vcs_checkout()  # noqa: auto unmount at the end of scope: on_execute()
            self.set_vcs_resulting_parameters()
            arc.initialize_arc_credentials(self.arc_token)
            initialize_aws_credentials(self.aws_key_id, self.aws_secret_key)
            initialize_yt_credentials(self.yt_token)

            ret_code = self.run_build_script()

            is_success = ret_code == 0

            if is_success:
                self.on_build_script_success()
            else:
                self.on_build_script_failure()

            try:
                self.on_build_script_finish(is_success)
            except Exception:
                self.logger.exception('on_build_script_finish got exception => ignored')

            if not is_success:
                failure_msg = 'CI scripts finished with non zero exit code({})'.format(ret_code)
                raise errors.TaskFailure(failure_msg)

            self.logger.info('Task {} finished'.format(self.__class__.__name__))

    def call(self, cmd, cwd=os.getcwd()):
        with sdk2.helpers.ProcessLog(self, logger=log) as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT, cwd=cwd)
