import json
import logging
import os

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.misc import NotExists
from sandbox.projects.sdc.common import session, utils
from sandbox.projects.sdc.common.constants import ROBOT_SDC_CI_SECRET_ID
from sandbox.projects.sdc.common_tasks.BaseSdcTask import BaseSdcTask
from sandbox.sdk2 import yav

import sandbox.common.types.client as ctc


class HitmanRunnerTask(BaseSdcTask):
    HITMAN_URL = 'https://hitman.yandex-team.ru'
    JOB_STATUS_OK = {'SUCCEEDED'}
    JOB_STATUS_FAILED = {'FAILED', 'CANCELED'}
    JOB_WAIT_TIME = 10 * 60  # 10 minutes for now

    class Requirements(BaseSdcTask.Requirements):
        cores = 1
        ram = 8 * 1024  # 8GB
        disk_space = 50 * 1024  # 50 GB
        client_tags = ctc.Tag.SSD

    def on_enqueue(self):
        pass

    def on_execute(self):
        with self.memoize_stage.run_hitman_process:
            with utils.create_venv() as venv:
                self.logger.info('Task {} started'.format(self.__class__.__name__))
                utils.configure_env(venv)
                self.git_checkout()
                self.run_build_script()
                hitman_status = os.path.join(str(self.get_working_dir()), 'artifacts', 'hitman_job_status.json')
                if not os.path.exists(hitman_status):
                    raise TaskFailure('Artifact with hitman job status wasnt created')
                with open(hitman_status) as f:
                    status = json.load(f)
                logging.info('Hitman process status: {}'.format(status))
                self.Context.job_id = str(status.get('jobId'))
                self.Context.nirvana_workflow_id = status.get('workflowId')

        if not self.wait_hitman_job_result(self.Context.job_id):
            raise TaskFailure('Hitman task failed or canceled')

    def on_break(self, prev_status, status):
        if self.Context.job_id != NotExists:
            logging.info('Trying to cancel hitman job {}'.format(self.Context.job_id))
            self.cancel_hitman_job(self.Context.job_id)
        else:
            logging.info('Hitman job wasnt started')

    @property
    def hitman_headers(self):
        secrets = yav.Yav(robot_sdc_ci=yav.Secret(ROBOT_SDC_CI_SECRET_ID),)
        return {
            'Content-Type': 'application/json',
            'Authorization': 'OAuth {}'.format(secrets.robot_sdc_ci['token.nirvana']),
        }

    @property
    def session(self):
        return session.create_session(retries=5, backoff_factor=2, backoff_max=30)

    def get_hitman_job(self, job_id):
        url = '/'.join([self.HITMAN_URL, 'api/v2/jobs', job_id])
        logging.info('Get Hitman job info, API url {}'.format(url))
        resp = self.session.get(url, headers=self.hitman_headers)
        logging.info('Hitman response: {}'.format(str(resp.content)))
        return resp.json()

    def get_job_status(self, job_id):
        return self.get_hitman_job(job_id)['status']

    def wait_hitman_job_result(self, job_id):
        status = self.get_job_status(job_id)
        logging.info("Job #{} status {}".format(job_id, status))
        if status in self.JOB_STATUS_OK:
            return True
        elif status in self.JOB_STATUS_FAILED:
            return False
        else:
            logging.info("Waiting for hitman job")
            raise sdk2.WaitTime(self.JOB_WAIT_TIME)

    def cancel_hitman_job(self, job_id):
        url = '/'.join([self.HITMAN_URL, 'api/v1/execution', job_id])
        resp = self.session.delete(url, headers=self.hitman_headers)
        logging.info('Hitman status code: {}'.format(str(resp.status_code)))
        resp.raise_for_status()
