import json
import logging
import os

from sandbox import common
from sandbox import sdk2
from sandbox.projects.sdc.common.constants import DEFAULT_VCS_TYPE
from sandbox.projects.sdc.common.constants import ROBOT_SDC_CI_SECRET_ID
from sandbox.projects.sdc.common.teamcity import TeamcityClient
from sandbox.projects.sdc.common.vcs_service import VcsService
from sandbox.projects.sdc.resource_types import SdcImportantLogs
from sandbox.common.types import client as ctc
from sandbox.sandboxsdk import environments
from sandbox.sdk2 import yav


class PrepareTask(sdk2.Task):
    BRANCH_RESOLVE_BUILD_CONF = 'Selfdriving_2_Docker_GetBranchFromRevision'
    BRANCH_ARTIFACT_PATH = 'branch.json'
    BRANCH_RESOLVE_BUILD_WAIT_TIME = 3 * 60

    LOGGER_IMPORTANT_LABEL = 'selfdriving-task-important-logs-for-investigation'
    LOGS_IMPORTANT_RES_PATH = 'important_logs.log'
    LOGGER_IMPORTANT = logging.getLogger(LOGGER_IMPORTANT_LABEL)

    class Parameters(sdk2.Task.Parameters):
        branch_or_commit = sdk2.parameters.String('Branch or commit', required=True)
        important_logs_parent_resource = sdk2.parameters.ParentResource(
            'SDC important logs parent resource', required=False
        )
        bb_server_url = sdk2.parameters.String('Bitbucket server base url')
        vcs_type = sdk2.parameters.String('VCS type', default=DEFAULT_VCS_TYPE)

    class Requirements(sdk2.Task.Requirements):
        environments = (environments.PipEnvironment('requests'),)
        client_tags = ctc.Tag.GENERIC

    def on_prepare(self):
        secrets = yav.Yav(
            robot_sdc_ci=yav.Secret(ROBOT_SDC_CI_SECRET_ID),
            # do not forget to delegate new tokens
        )
        self.arc_token = secrets.robot_sdc_ci['token.arc']
        self.arcanum_token = secrets.robot_sdc_ci['token.arcanum']
        self.aws_key_id = secrets.robot_sdc_ci['aws.key_id']
        self.aws_secret_key = secrets.robot_sdc_ci['aws.secret']
        self.bb_token = secrets.robot_sdc_ci['token.bb']
        self.teamcity_token = secrets.robot_sdc_ci['token.teamcity']

        self.vcs_service = VcsService(vcs_type=self.Parameters.vcs_type,
                                      arc_token=self.arc_token,
                                      arcanum_token=self.arcanum_token,
                                      bb_token=self.bb_token,
                                      bb_server_url=self.Parameters.bb_server_url)

    def _teamcity_client(self):
        return TeamcityClient.with_oauth(self.teamcity_token)

    def _contains_branch(self):
        try:
            branch_or_commit = self.Parameters.branch_or_commit
            # svn revision
            if self.Parameters.branch_or_commit.isdecimal():
                branch_or_commit = 'r' + branch_or_commit
            return self.vcs_service.resolve_branch_and_commit(branch_or_commit)
        except Exception as e:
            self.LOGGER_IMPORTANT.exception(
                'Cannot resolve branch: %s. %s: {}',
                self.Parameters.branch_or_commit, repr(e))
            raise common.errors.TaskFailure('Cannot resolve branch')

    def _get_branch_name_from_task(self, branch_resolve_task_id):
        try:
            self._download_artifact(branch_resolve_task_id, self.BRANCH_ARTIFACT_PATH)
            with open(os.path.basename(self.BRANCH_ARTIFACT_PATH)) as f:
                branch_map = json.load(f)
            return branch_map['branch']
        except Exception as e:
            self.LOGGER_IMPORTANT.exception(
                'Cannot get branch name from task %s: %s. Path: {}',
                branch_resolve_task_id, repr(e), self.BRANCH_ARTIFACT_PATH
            )
            raise common.errors.TaskFailure(
                'Cannot get branch name from task {}'.format(branch_resolve_task_id))

    def _download_artifact(self, build_id, artifact_path):
        try:
            tc_client = self._teamcity_client()
            tc_client.download_artifact(build_id, artifact_path)
        except Exception as e:
            self.LOGGER_IMPORTANT.exception('Cannot download artifact %s from build %s: %s. %s',
                                            artifact_path, build_id, repr(e), TeamcityClient.artifact_url(build_id, artifact_path))
            raise Exception('Cannot download artifact {} from {}'.format(artifact_path, build_id))

    def _get_build_details(self, build_id):
        try:
            tc_client = self._teamcity_client()
            return tc_client.build_details(build_id)
        except Exception as e:
            self.LOGGER_IMPORTANT.exception('Cannot get %s build details: %s. %s', build_id, repr(e), TeamcityClient.build_url(build_id))
            raise Exception('Cannot get {} build_details'.format(build_id))

    def _get_build_resulting_props(self, build_id):
        try:
            tc_client = self._teamcity_client()
            return tc_client.build_resulting_props(build_id)
        except Exception as e:
            self.LOGGER_IMPORTANT.exception('Cannot get %s build resulting props: %s. %s', build_id, repr(e), TeamcityClient.build_url(build_id))
            raise Exception('Cannot get {} build_resulting_props'.format(build_id))

    def _trigger_build_on_teamcity(self, build_conf, branch=None, last_changes=None, params=None):
        try:
            tc_client = self._teamcity_client()
            tree = tc_client.trigger_build(build_conf, branch, last_changes, params)
            web_url = tree.attrib['webUrl']
            self.LOGGER_IMPORTANT.debug('Triggered build %s: %s', build_conf, web_url)
            return tree.attrib['id']
        except Exception as e:
            self.LOGGER_IMPORTANT.exception(
                'Cannot trigger build %s for branch %s, changes %s, params %s: %s', build_conf, branch, last_changes, params, repr(e))
            raise common.errors.TaskFailure('Cannot trigger build {}'.format(build_conf))

    def _check_teamcity_task_status(self, build_id, wait_timeout):
        try:
            build_tree = self._get_build_details(build_id)
            if build_tree.attrib['state'] != 'finished':
                raise sdk2.WaitTime(wait_timeout)
            elif build_tree.attrib['status'] != 'SUCCESS':
                self.LOGGER_IMPORTANT.error('Build status is not SUCCESS, %s', build_tree.attrib['webUrl'])
                raise Exception('Build {} failed'.format(build_id))
        except Exception as e:
            self.LOGGER_IMPORTANT.exception(
                'Cannot check build %s status: %s. %s', build_id, repr(e), TeamcityClient.build_url(build_id))
            raise common.errors.TaskFailure('Cannot check build {} status or build failed'.format(build_id))

    def on_finish(self, prev_status, status):
        try:
            log_resources_count = sdk2.Resource.find(task=self, type='TASK_LOGS').count
            log_resources = sdk2.Resource.find(task=self, type='TASK_LOGS')

            if not log_resources:
                return

            log_important_data = ''
            for r in log_resources.limit(log_resources_count):
                rd = sdk2.ResourceData(r)
                log_data = rd.path.joinpath("debug.log").read_text()
                for l in log_data.splitlines():
                    marker_full = self.LOGGER_IMPORTANT_LABEL  # there is no more () in sandbox log
                    if marker_full in l:
                        log_important_data += l.replace(marker_full + ' ', '', 1) + '\n'

            if not log_important_data:
                return

            if not self.Parameters.important_logs_parent_resource:
                with open(self.LOGS_IMPORTANT_RES_PATH, 'w') as f:
                    f.write(log_important_data)
                important_logs_resource = SdcImportantLogs(self, 'Sdc important logs', self.LOGS_IMPORTANT_RES_PATH, ttl=1)
                sdk2.ResourceData(important_logs_resource).ready()
            else:
                # todo(gorshkovna): i cant use it through several tasks. it has to be direct parent
                logs_data = sdk2.ResourceData(self.Parameters.important_logs_parent_resource)
                logs_path = logs_data.path
                if not logs_path.exists():
                    logs_path.touch()
                logs_path.write_text(log_important_data)
                logs_data.ready()
        except Exception as e:
            msg = 'Something went wrong while finishing task'
            self.LOGGER_IMPORTANT.exception('{}, %s'.format(msg), repr(e))
            raise common.errors.TaskFailure(msg)
