import logging
import time

import requests

from sandbox.projects.common import binary_task
from sandbox import sdk2

from sandbox.projects.common import task_env


class SdcIcReleaseWorkers(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(sdk2.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        sandbox_token = sdk2.parameters.YavSecret(
            "YAV secret identifier (with optional version)",
            required=True,
            default="sec-01ejdt9bs00fnnmjan81b9q6j1",
        )
        isolate_cloud_environment = sdk2.parameters.String(
            'IsolateCloud environment (testing or production)',
            required=True,
            default='testing',
        )
        worker_groups = sdk2.parameters.List(
            'Worker groups to update (will be updated sequentially)',
            required=True
        )
        nanny_service_name = sdk2.parameters.String(
            'Nanny service that will be used to determine which Sandbox resource with binaries is currently deployed.'
            ' Optional, will be inferred based on IsolateCloud environment.',
            required=False,
            default='',
        )
        resource_with_binaries_type = sdk2.parameters.String(
            'Type of the Sandbox resource that contains binaries, is deployed to the Nanny service and has'
            ' a "commit" attribute',
            required=True,
            default='SDC_BINARIES_ARCHIVE',
        )
        max_minutes_per_group = sdk2.parameters.Integer(
            'Maximum time per worker group in minutes',
            required=True,
            default=90,
        )

    def on_execute(self):
        from sdg.library.python.isolate_cloud.client import IsolateCloudClient

        ic_client = IsolateCloudClient.create_by_instance_name(
            instance_name=self.Parameters.isolate_cloud_environment,
            oauth_token=self.Parameters.sandbox_token.data()['ISOLATE_CLOUD_TOKEN'],
        ).create_workers_api_client()

        binaries_resource_id = self._get_binaries_resource_id_from_nanny_service()
        binaries_commit_id = self._get_binaries_commit_id_from_resource(binaries_resource_id=binaries_resource_id)
        for worker_group_id_comma_separated in self.Parameters.worker_groups:
            for worker_group_id in worker_group_id_comma_separated.split(','):
                self._release_binaries_commit_id_to_worker_group(
                    ic_client=ic_client,
                    binaries_commit_id=binaries_commit_id,
                    worker_group_id=worker_group_id,
                )
                self._wait_for_worker_group_to_activate(
                    ic_client=ic_client,
                    worker_group_id=worker_group_id,
                )

    def _get_binaries_resource_id_from_nanny_service(self):
        nanny_service_name = self._get_nanny_service_name()
        logging.info('Getting binaries_resource_id from Nanny service %s', nanny_service_name)
        response = requests.get(
            url='https://nanny.yandex-team.ru/v2/services/{}/runtime_attrs/resources/'.format(nanny_service_name),
        )
        response.raise_for_status()
        for sandbox_file in response.json()['content']['sandbox_files']:
            if sandbox_file['resource_type'] == self.Parameters.resource_with_binaries_type:
                return sandbox_file['resource_id']
        raise Exception("Nanny service {} doesn't contain resource {}"
                        "".format(nanny_service_name, self.Parameters.resource_with_binaries_type))

    def _get_nanny_service_name(self):
        return 'sdc_isolate_cloud_yt_operations_runner_{}'.format(self.Parameters.isolate_cloud_environment)

    def _get_binaries_commit_id_from_resource(self, binaries_resource_id):
        logging.info('Getting binaries_commit_id from resource %s', binaries_resource_id)
        response = requests.get(
            url='https://sandbox.yandex-team.ru/api/v1.0/resource/{}'.format(binaries_resource_id),
            headers={
                'Authorization':
                    'OAuth {}'.format(self.Parameters.sandbox_token.data()['SANDBOX_TOKEN'])
            },
        )
        response.raise_for_status()
        return response.json()['attributes']['commit']

    def _release_binaries_commit_id_to_worker_group(self, ic_client, binaries_commit_id, worker_group_id):
        logging.info('Releasing binaries %s to worker group %s', binaries_commit_id, worker_group_id)
        response = ic_client.get_worker_group(group_id=worker_group_id)
        response.raise_for_status()
        version = response.json()
        if version.get('worker_config', {}).get('binaries_commit') == binaries_commit_id:
            return
        version['previous_version_id'] = version['current_version_id']
        del version['current_version_id']
        version['worker_config']['binaries_commit'] = binaries_commit_id
        response = ic_client.create_new_worker_group_version(group_id=worker_group_id, version=version)
        response.raise_for_status()

    def _wait_for_worker_group_to_activate(self, ic_client, worker_group_id):
        logging.info('Waiting for worker group %s to activate', worker_group_id)
        start_time = time.time()
        while True:
            if self._worker_group_is_active(ic_client=ic_client, worker_group_id=worker_group_id):
                logging.info('Worker group %s is active', worker_group_id)
                return
            if time.time() - start_time > 60 * self.Parameters.max_minutes_per_group:
                raise Exception("Group {} failed to activate after {} minutes"
                                "".format(worker_group_id, self.Parameters.max_minutes_per_group))
            logging.info("Worker group %s isn't active, waiting", worker_group_id)
            time.sleep(5)

    def _worker_group_is_active(self, ic_client, worker_group_id):
        response = ic_client.get_worker_groups()
        response.raise_for_status()
        for worker_group in response.json()['items']:
            if worker_group['group_id'] == worker_group_id:
                return (worker_group['worker_count'] == 0 or
                        worker_group['last_version_alive_workers_count'] / worker_group['worker_count'] >= 0.95)
        raise Exception("Worker group {} doesn't exist".format(worker_group_id))
