import logging

from sandbox.common.types import resource as ctr
from sandbox import sdk2

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm


class SdcIcResizeWorkerGroups(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 8 * 1024  # 8 GB
        disk_space = 32 * 1024  # 32 GB
        client_tags = ctc.Tag.SSD
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass  # To execute on MULTISLOT clients, https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot

    class Parameters(sdk2.Parameters):
        instance_name = sdk2.parameters.String(
            'Use "prod" or "testing"', required=True, default='testing',
        )
        worker_count_by_group = sdk2.parameters.Dict("Worker groups and new sizes")
        token = sdk2.parameters.YavSecret(
            "YAV secret identifier (with optional version)",
            required=True,
            default="sec-01ejdt9bs00fnnmjan81b9q6j1",
        )

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"tasks_bundle": "TEST_TASKS_BUNDLE", "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def on_execute(self):
        from sdg.library.python.isolate_cloud.client import IsolateCloudClient

        oauth_token = self.Parameters.token.data()['ISOLATE_CLOUD_TOKEN']

        if not oauth_token.startswith('AQAD'):
            raise Exception('Prefix is {}'.format(str(oauth_token)[:4]))

        client = IsolateCloudClient.create_by_instance_name(
            instance_name=self.Parameters.instance_name,
            oauth_token=oauth_token,
        ).create_workers_api_client()

        for group_id, worker_count in self.Parameters.worker_count_by_group.items():
            logging.info('About to resize group {} to {} workers ... '.format(group_id, worker_count))
            response = client.resize_group(group_id, worker_count)
            logging.info(response.text)
            response.raise_for_status()

        logging.info('All groups were resized!')
