import logging

import sandbox.common.types.task as ctt

from sandbox import common
from sandbox import sdk2
from sandbox.projects.sdc.common import constants
from sandbox.projects.sdc.common.constants import SANDBOX_RESOURCE_ATTRIBUTE
from sandbox.projects.sdc.common.constants import ROBOT_SDC_CI_SECRET_ID
from sandbox.projects.sdc.common.yt_helpers import create_clusters, get_cluster_unavailable_errors, YT_VERSION
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.sdc.resource_types import SdcImportantLogs
from sandbox.sdk2 import yav


class AbstractSdcBuildPortoLayer(sdk2.Task):
    IMPORTANT_LOGS_PATH = 'important_logs.log'

    class Requirements(sdk2.Requirements):
        environments = (
            PipEnvironment('yandex-yt', YT_VERSION),
        )

    class Parameters(sdk2.Parameters):
        branch = sdk2.parameters.String('Branch', required=False)
        commit = sdk2.parameters.String('Commit', required=True)
        clusters = sdk2.parameters.List('Clusters to upload layers', required=True)
        yt_path = sdk2.parameters.String('YT path for porto layer', required=True)
        important_logs_parent_resource = sdk2.parameters.ParentResource(
            'SDC important logs parent resource', required=False
        )
        bb_server_url = sdk2.parameters.String('Bitbucket server base url')
        vcs_type = sdk2.parameters.String('VCS type', default=constants.DEFAULT_VCS_TYPE)

    def on_enqueue(self):
        # acquire semaphore to make sure, that we are building one porto layer for every version
        layer_filename = self.Parameters.yt_path.split('/')[-1]
        self.Context.layer_name, self.Context.compress = layer_filename.split('.', 1)
        self.Context.ttl = 14
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[ctt.Semaphores.Acquire(name=self.get_semaphore_name(), capacity=1)],
            release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        )

    def get_semaphore_name(self):
        return self.Context.layer_name

    def on_prepare(self):
        secrets = yav.Yav(
            robot_sdc_ci=yav.Secret(ROBOT_SDC_CI_SECRET_ID),
        )
        self.bb_token = secrets.robot_sdc_ci['token.bb']
        self.yt_token = secrets.robot_sdc_ci['token.yt']
        self.clusters = create_clusters(self.Parameters.clusters, self.yt_token)
        self.cluster_unavailable_exceptions = get_cluster_unavailable_errors()

    def check_for_success(self, task_id):
        if sdk2.Task[task_id].status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure('Error in task {}'.format(task_id))

    def create_preparation_task(self, sdc_important_logs_resource):
        raise NotImplementedError()

    def build_porto_layer(self):
        raise NotImplementedError()

    def get_porto_layer_filename(self):
        return '{}.{}'.format(self.Context.layer_name, self.Context.compress)

    def check_existence(self, layer_yt_path):
        absent_in_clusters = []
        for yt_cluster in self.clusters:
            try:
                if not yt_cluster.client.exists(layer_yt_path):
                    absent_in_clusters.append(yt_cluster)
            except self.cluster_unavailable_exceptions as e:
                # ignore connection error due to cluster unavailability,
                # assuming, that only one cluster can be unavailable at the time
                logging.info('Cluster {} is unavailable'.format(yt_cluster.name))
                logging.exception(e)
        return absent_in_clusters

    def upload_layer_to_yt(self, local_layer_path, clusters, commit, branch, resource_id):
        # we don't have to lock path in YT, because this task acquires semaphore for this path
        for cluster in clusters:
            try:
                yt = cluster.client
                yt_path = self.Parameters.yt_path
                with yt.Transaction():
                    yt.create('file', yt_path, recursive=True, ignore_existing=True)
                    with open(local_layer_path) as f:
                        yt.write_file(yt_path, f)
                    yt.set_attribute(yt_path, 'commit', commit)
                    yt.set_attribute(yt_path, 'branch', branch)
                    yt.set_attribute(yt_path, SANDBOX_RESOURCE_ATTRIBUTE, resource_id)
            except self.cluster_unavailable_exceptions as e:
                # ignore connection error due to cluster unavailability,
                # assuming, that only one cluster can be unavailable at the time
                logging.info('Cluster {} is unavailable'.format(cluster.name))
                logging.exception(e)

    def copy(self, from_cluster, to_clusters):
        logging.info('Copying layer from {} to {}'.format(from_cluster, to_clusters))
        try:
            yt = from_cluster.client
            layer = yt.read_file(self.Parameters.yt_path)
            layer_filename = self.get_porto_layer_filename()
            with open(layer_filename, 'w') as f:
                f.write(layer.read())
            # we copy attributes from original layer, because this task could be triggered on another commit
            layer_info = yt.get(self.Parameters.yt_path, attributes=['commit', 'branch', SANDBOX_RESOURCE_ATTRIBUTE])
            self.upload_layer_to_yt(
                layer_filename, to_clusters, layer_info.attributes['commit'],
                layer_info.attributes['branch'], layer_info.attributes[SANDBOX_RESOURCE_ATTRIBUTE]
            )
            return True
        except self.cluster_unavailable_exceptions as e:
            # ignore connection error due to cluster unavailability,
            # assuming, that only one cluster can be unavailable at the time
            logging.info('Cluster {} is unavailable'.format(from_cluster.name))
            logging.exception(e)
            return False

    def porto_layer_exists(self):
        absent_in_clusters = self.check_existence(self.Parameters.yt_path)
        # if yt_path exists in all clusters, we don't need to build or copy
        if len(absent_in_clusters) == 0:
            return True
        # copy layer to clusters in which it's absent, so we don't need to build it again
        # todo: if one of clusters is unavailable, than here we will try to copy,
        #  which isn't correct. Maybe I need to check cluster availability on task startup,
        #  and then use it as assumption for the remaining code. It will make code more clear
        if not len(absent_in_clusters) == len(self.Parameters.clusters):
            present_in_clusters = list(set(self.clusters) - set(absent_in_clusters))
            if self.copy(present_in_clusters[0], absent_in_clusters):
                return True
        return False

    def on_execute(self):
        with self.memoize_stage.check_existence:
            # we need to check whether yt path already exists in all clusters, because
            # we could have one task building layers and one task waiting on semaphore
            if self.porto_layer_exists():
                return

        with self.memoize_stage.prepare_porto_layer:
            if self.Parameters.important_logs_parent_resource:
                sdc_important_logs_resource = self.Parameters.important_logs_parent_resource
            else:
                sdc_important_logs_resource = SdcImportantLogs(self, '', self.IMPORTANT_LOGS_PATH)
            preparation_task = self.create_preparation_task(sdc_important_logs_resource)
            self.Context.preparation_task_id = preparation_task.enqueue().id
            raise sdk2.WaitTask(
                [self.Context.preparation_task_id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )

        self.build_porto_layer()
