# coding=utf-8
from __future__ import unicode_literals

import logging
import os.path
import re
from datetime import date

import aniso8601
import requests

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import task_env
from sandbox.projects.common.decorators import retries
from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.sdc.common import arc
from sandbox.projects.sdc.common.arc import ARC_SDC_REL_PATH
from sandbox.projects.sdc.resource_types import SdcBinariesArchive
from sandbox.sdk2.environments import PipEnvironment

SECRET_ID = 'sec-01e47j727mj21xbag7hq8h1rt5'


class CommitInfo:
    def __init__(self, datetime, hash, message, author, pr, alter):
        self.datetime = datetime
        self.hash = hash
        self.message = re.sub(r'\s*REVIEW: \d+', '', message)
        self.author = author
        self.pr = pr
        self.alter = alter

    @property
    def wiki_row(self):
        return '||{}||'.format('|'.join([
            self.datetime.strftime('%Y‑%m‑%d&nbsp;%H:%M:%S'),
            '((https://a.yandex-team.ru/arc_vcs/commit/{0} {0}))'.format(self.hash),
            self.message.replace('|', r'\|'),
            '@{}'.format(self.author),
            '((https://a.yandex-team.ru/review/{0} {0}))'.format(self.pr)
        ]))


class SdcSimulatorRelease(binary_task.LastRefreshableBinary, sdk2.Task):
    ARC_LOG_PATH = os.path.join(ARC_SDC_REL_PATH, 'simulator/backend')
    ALTERS_FILE = 'sdg/sdc/simulator/backend/simdb/simstate.py'

    class Requirements(task_env.TinyRequirements):
        environments = [
            PipEnvironment('startrek-client', '2.5', custom_parameters=['--upgrade-strategy only-if-needed']),
        ]

    class Parameters(sdk2.Parameters):
        revision = sdk2.parameters.Integer('Release revision', description='Release revision')
        arc_token = sdk2.parameters.YavSecret(
            'Arc oauth token', default='{}#ARC_TOKEN'.format(SECRET_ID), required=True
        )
        abc_token = sdk2.parameters.YavSecret(
            'ABC oauth token', default='{}#ABC_TOKEN'.format(SECRET_ID), required=True
        )
        st_token = sdk2.parameters.YavSecret(
            'ST oauth token', default='{}#ST_TOKEN'.format(SECRET_ID), required=True
        )
        tsum_session_id_cookie = sdk2.parameters.YavSecret(
            'TSUM sessionid2 cookie', default='{}#sessionid2'.format(SECRET_ID), required=True
        )

        _binary = binary_task.binary_release_parameters_list(none=True)

    def on_save(self):
        binary_task.LastRefreshableBinary.on_save(self)
        sdk2.Task.on_save(self)

    def on_enqueue(self):
        binary_task.LastRefreshableBinary.on_enqueue(self)
        sdk2.Task.on_enqueue(self)

    def on_wait(self, prev_status, status):
        binary_task.LastRefreshableBinary.on_wait(self, prev_status, status)
        sdk2.Task.on_wait(self, prev_status, status)

    def create_issue_description(self, commits_info, alters):
        if alters:
            alters_block = """\
%%(wacko wrapper=shade width=750 border="1px solid red")
Не забудьте применить альтеры:
{}
%%""".format('\n'.join([
                '{2}. ((https://a.yandex-team.ru/arc_vcs/commit/{0}#file-{1} {0}))'.format(alter, self.ALTERS_FILE, i + 1)
                for i, alter in enumerate(alters)
            ]))

        else:
            alters_block = ''

        commits_block = """\
#|
||**Date**|**Hash**|**Message**|**Author**|**Review**||
{}
|#""".format('\n'.join([ci.wiki_row for ci in commits_info]))

        return '\n'.join([block for block in [alters_block, commits_block] if block])

    @retries(3)
    def get_backend_duty(self):
        resp = requests.get(
            'https://abc-back.yandex-team.ru/api/v4/duty/on_duty/?schedule__slug=web-simulator-backend',
            headers={'Authorization': 'OAuth {}'.format(self.Parameters.abc_token.data()[self.Parameters.abc_token.default_key])}
        )
        resp.raise_for_status()
        return resp.json()[0]['person']['login']

    @retries(3)
    def get_tsum_pipeline(self):
        PIPE_PREFIX = 'PIPE-'
        for tag in self.Parameters.tags:
            if tag.startswith(PIPE_PREFIX):
                pipeline_id = tag[len(PIPE_PREFIX):].lower()
                break
        else:
            raise Exception('Pipeline tag not found')

        resp = requests.get(
            'https://tsum.yandex-team.ru/api/projects/sdc-web-simulator/active-releases',
            cookies={
                'sessionid2': self.Parameters.tsum_session_id_cookie.data()[self.Parameters.tsum_session_id_cookie.default_key],
            }
        )
        resp.raise_for_status()
        try:
            releases = resp.json()
        except Exception:
            logging.debug(resp.text)
            raise

        for release in releases:
            if release['pipeId'] == 'backend-cd' and pipeline_id in release['pipeLaunchIds']:
                return release
        else:
            raise Exception('Release not found for pipeline {}'.format(pipeline_id))

    def on_execute(self):
        from startrek_client import Startrek

        duty = self.get_backend_duty()

        last_released_resource = SdcBinariesArchive.find(
            attrs={'released': 'stable',  'program': 'simbackend'},
        ).first()
        if not last_released_resource:
            raise Exception('Last released resource for program {} not found'.format('simbackend'))

        stable_commit = last_released_resource.commit
        arc_cli = Arc(arc_oauth_token=self.Parameters.arc_token.data()[self.Parameters.arc_token.default_key])

        if 'TSUM' in self.Parameters.tags:
            pipeline = self.get_tsum_pipeline()
            release_author = pipeline['triggeredBy']
            revision = pipeline['revision']
        elif not self.Parameters.revision:
            raise Exception('Revision is required for non-TSUM launches')
        else:
            if self.author != 'robot-simulator':
                release_author = self.author
            else:
                release_author = duty
            revision = self.Parameters.revision

        release_commit = arc.get_commit(arc_cli, 'r{}'.format(revision))
        if not release_commit:
            raise Exception('Commit not found for revision {}'.format(revision))

        with arc_cli.init_bare() as mp:
            commits = arc_cli.log(
                mp,
                path=self.ARC_LOG_PATH,
                start_commit=stable_commit,
                end_commit=release_commit,
                name_only=True,
                as_dict=True,
            )
        alter_commits = []
        commits_info = []
        for commit_info in commits:
            files = {e['path'] for e in commit_info['names']}
            if self.ALTERS_FILE in files:
                alter_commits.append(commit_info['commit'])
            commits_info.append(CommitInfo(
                datetime=aniso8601.parse_datetime(commit_info['date']),
                hash=commit_info['commit'],
                author=commit_info['author'],
                message=commit_info['message'],
                pr=commit_info['attributes']['pr.id'],
                alter=bool(self.ALTERS_FILE in files)
            ))

        logging.getLogger('yandex_tracker_client.objects').setLevel(logging.INFO)
        st_client = Startrek(token=self.Parameters.st_token.data()[self.Parameters.st_token.default_key], useragent='')
        issue = st_client.issues.create(
            queue='WEBSIM', type='task',
            assignee=release_author,
            followers=list({release_author, duty, 'albelyaev'}),
            tags=['simbackend_release'],
            summary='Релиз бэкенда симулятора {}'.format(date.today().strftime('%d.%m.%Y')),
            description=self.create_issue_description(commits_info, alter_commits),
            links=[{
                'relationship': 'relates',
                'origin': 'ru.yandex.sandbox',
                'key': self.id
            }]
        )
        self.set_info('{0}: <a href="https://st.yandex-team.ru/{1}">{1}</a>'.format('Релизный тикет', issue.key), do_escape=False)
