# coding=utf-8
from __future__ import unicode_literals

import logging
import re

from pathlib2 import Path

from sandbox import sdk2
from sandbox.common.patterns import singleton_property
from sandbox.common.types.task import TaskStatus
from sandbox.projects.common import binary_task
from sandbox.projects.common import task_env
from sandbox.projects.common.vcs.arc import ArcCommandFailed
from sandbox.projects.metrika.utils.parameters import choices
from sandbox.projects.sdc.common import constants, arc
from sandbox.projects.sdc.common.vcs_service import VcsService
from sandbox.projects.sdc.simulator.SdcSimulatorRunRegular import SdcSimulatorRunRegular
from sandbox.projects.sdc.simulator.SdcSimulatorRunRegularity import SdcSimulatorRunRegularity
from sandbox.projects.sdc.simulator.utils import SECRET_ID


class SdcSimulatorReleaseProgram(binary_task.LastRefreshableBinary, sdk2.Task):
    COMMIT_PATTERN = "^STABLE_COMMIT = '(?P<commit>.*?)'$"

    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(sdk2.Parameters):
        program = sdk2.parameters.String('Program', choices=choices([
            SdcSimulatorRunRegularity.__name__,
            SdcSimulatorRunRegular.__name__,
        ]), required=True, default=SdcSimulatorRunRegularity.__name__)
        revision = sdk2.parameters.Integer('Revision', required=True)

        arc_token = sdk2.parameters.YavSecret('Arc oauth token', default='{}#ARC_TOKEN'.format(SECRET_ID))

        _binary = binary_task.binary_release_parameters_list(none=True)

    def on_save(self):
        binary_task.LastRefreshableBinary.on_save(self)
        sdk2.Task.on_save(self)

    def on_enqueue(self):
        binary_task.LastRefreshableBinary.on_enqueue(self)
        sdk2.Task.on_enqueue(self)

    def on_wait(self, prev_status, status):
        binary_task.LastRefreshableBinary.on_wait(self, prev_status, status)
        sdk2.Task.on_wait(self, prev_status, status)

    @singleton_property
    def arc_token(self):
        return self.Parameters.arc_token.data()[self.Parameters.arc_token.default_key]

    @singleton_property
    def arc_cli(self):
        return arc.build_client(self.arc_token)

    def on_execute(self):
        with self.memoize_stage.create_pr(commit_on_entrance=False):
            self.create_pr()
        if self.Context.wait:
            self.wait_pr()

    def checkout(self):
        return self.arc_cli.mount_path(
            path=None,
            changeset='trunk',
            fetch_all=False
        )

    def create_pr(self):
        commit = VcsService(vcs_type=constants.VCS_ARC, arc_token=self.arc_token).get_commit_from_branch_or_commit(
            'r{}'.format(self.Parameters.revision)
        )

        with self.checkout() as arcadia_root:
            program_source_file = Path(arcadia_root).joinpath('sandbox/projects/sdc/simulator', self.Parameters.program, '__init__.py')
            if not program_source_file.exists():
                raise Exception('Source file doesn\'t exist for program {}'.format(self.Parameters.program))

            program_source = program_source_file.read_text(encoding='utf8')
            m = re.search(self.COMMIT_PATTERN, program_source, re.MULTILINE)
            if not m:
                raise Exception('Source file doesn\'t contain STABLE_COMMIT variable')

            if m.group('commit') == commit:
                logging.info('Trunk version already contains this commit: %s', commit)
                self.Context.wait = False
                return
            else:
                self.Context.wait = True

            branch = '{}-bump-{}'.format(self.Parameters.program, commit)
            self.arc_cli.checkout(arcadia_root, branch, create_branch=True)
            try:
                self.arc_cli.branch(arcadia_root, upstream='users/robot-simulator/{}'.format(branch))
            except ArcCommandFailed as e:
                if 'no such remote branch' not in str(e):
                    raise

            try:
                self.Context.pr_id = self.arc_cli.pr_status(arcadia_root, as_dict=True)['id']
            except ArcCommandFailed as e:
                if 'Can\'t find any PR for current branch' in str(e):
                    program_source = re.sub(self.COMMIT_PATTERN, "STABLE_COMMIT = '{}'".format(commit), program_source, flags=re.MULTILINE)
                    program_source_file.write_text(program_source, encoding='utf8')
                    self.arc_cli.commit(arcadia_root, 'bump {} to {}'.format(self.Parameters.program, commit), all_changes=True)
                    self.arc_cli.push(arcadia_root, force=True)
                    self.Context.pr_id = self.arc_cli.pr_create(arcadia_root, auto=True, no_edit=True, no_push=True, as_dict=True)['id']
                else:
                    raise

            self.set_info('<a href="https://a.yandex-team.ru/review/{}">PR</a>'.format(self.Context.pr_id), do_escape=False)

    def wait_pr(self):
        logging.info('Waiting for pr %s', self.Context.pr_id)
        with self.checkout() as arcadia_root:
            pr_info = self.arc_cli.pr_status(arcadia_root, self.Context.pr_id, as_dict=True)
            if pr_info['status'] != 'merged':
                raise sdk2.WaitTime(60 * 5)

    def on_break(self, prev_status, status):
        binary_task.LastRefreshableBinary.on_wait(self, prev_status, status)

        if status == TaskStatus.STOPPED and self.Context.pr_id:
            logging.info('Discarding pr %s', self.Context.pr_id)
            with self.checkout() as arcadia_root:
                try:
                    self.arc_cli.pr_discard(arcadia_root, self.Context.pr_id)
                except ArcCommandFailed as e:
                    if 'already closed' in str(e):
                        logging.info(str(e))
                    else:
                        raise

        sdk2.Task.on_break(self, prev_status, status)
