# coding=utf-8
from __future__ import unicode_literals

import filecmp
import json
import logging
import re
import shutil

from pathlib2 import Path
from sandbox.projects.resource_types import OTHER_RESOURCE

from sandbox import sdk2
from sandbox.common.errors import TemporaryError
from sandbox.common.patterns import singleton_property
from sandbox.common.types.task import Status
from sandbox.common.urls import get_task_link
from sandbox.projects.common import binary_task
from sandbox.projects.common import task_env
from sandbox.projects.metrika.utils.mixins.subtasks import SubtasksMixin
from sandbox.projects.sdc.SdcBuildNoTests import SdcBuildNoTests
from sandbox.projects.sdc.common import constants, arc
from sandbox.projects.sdc.common.arc import get_commits
from sandbox.projects.sdc.common.aws import initialize_aws_credentials
from sandbox.projects.sdc.common.docker import login_to_docker, configure_docker
from sandbox.projects.sdc.common.parameters import LastLxcContainerResource
from sandbox.projects.sdc.common.utils import run_command, RunCommandException
from sandbox.projects.sdc.common.vcs_service import VcsService
from sandbox.projects.sdc.common.yt_helpers import initialize_yt_credentials
from sandbox.projects.sdc.simulator.utils import get_binaries_url, binaries_are_ready, SECRET_ID

STABLE_COMMIT = 'trunk'


class CommonParameters(sdk2.Parameters):
    container = LastLxcContainerResource('Last LXC resource', required=False)
    retries = sdk2.parameters.Integer('Retries count', default=0)

    with sdk2.parameters.Group('Program binaries') as binaries_group:
        use_stable = sdk2.parameters.Bool('Use stable program commit', default=False, description=STABLE_COMMIT)
        with use_stable.value[False]:
            binaries_branch_or_commit = sdk2.parameters.String('Binaries branch or commit hash', default='trunk')
        recent_builded = sdk2.parameters.Bool('Take builded binaries from most recent commit on branch', default=True)

    with sdk2.parameters.Group('Output') as output_group:
        program_output_path = sdk2.parameters.String('Relative to SDC_ROOT file/dir path with output')
        output_as_parameter = sdk2.parameters.Bool('Save output in output parameter instead of resource', default=True)

        with output_as_parameter.value[True]:
            output_is_json = sdk2.parameters.Bool('Output is JSON', default=False)
            with sdk2.parameters.Output(reset_on_restart=True):
                with output_is_json.value[True]:
                    json_output = sdk2.parameters.JSON('Program JSON output')

                with output_is_json.value[False]:
                    output = sdk2.parameters.String('Program output')

    _binary = binary_task.binary_release_parameters_list(none=True)

    with sdk2.parameters.Group('Task development parameters (don\'t change)') as secret_params:
        show_secret_params = sdk2.parameters.Bool(
            'Show task development parameters', default=False, description='Yes, I know what i\'m doing'
        )
        with show_secret_params.value[True]:
            with sdk2.parameters.Group('Arc') as arc_group:
                use_sources = sdk2.parameters.Bool(
                    'Use sources from repo', default=False,
                    description='Checkout repo, use it first and then binaries'
                )
                with use_sources.value[True]:
                    sources_branch = sdk2.parameters.String(
                        'Sources branch', required=True, default='trunk'
                    )

                arc_token = sdk2.parameters.YavSecret('Arc oauth token', default='{}#ARC_TOKEN'.format(SECRET_ID))

            with sdk2.parameters.Group('Docker') as docker_group:
                docker_login = sdk2.parameters.String('Docker login', default='robot-simulator')
                docker_token = sdk2.parameters.YavSecret('Docker oauth token', default='{}#DOCKER_TOKEN'.format(SECRET_ID))

            with sdk2.parameters.Group('YT') as yt_group:
                yt_token = sdk2.parameters.YavSecret('YT oauth token', default='{}#YT_TOKEN'.format(SECRET_ID))

            with sdk2.parameters.Group('MDS') as mds_group:
                aws_access_key_id = sdk2.parameters.YavSecret('AWS oauth token', default='{}#AWS_ACCESS_KEY_ID'.format(SECRET_ID))
                aws_secret_access_key = sdk2.parameters.YavSecret('AWS oauth token', default='{}#AWS_SECRET_ACCESS_KEY'.format(SECRET_ID))


class SdcSimulatorRunProgram(binary_task.LastRefreshableBinary, sdk2.Task, SubtasksMixin):
    STABLE_COMMIT = STABLE_COMMIT

    class Requirements(task_env.BuildRequirements):
        cores = 4
        ram = 32 * 1024
        disk_space = 32 * 1024

    class Context(sdk2.Context):
        docker_retries = 0
        commit = None
        failed = None
        started = False

    class Parameters(CommonParameters):
        program = sdk2.parameters.String('Command to run in docker', required=True, description='./d --run "{program}"', multiline=True)
        with sdk2.parameters.Group('Env Vars') as env_vars_group:
            env_vars = sdk2.parameters.Dict('Env Vars', description='sec-id#key for secrets')

    @singleton_property
    def sdc_root(self):
        return self.path('sdc')

    @singleton_property
    def arc_token(self):
        return self.Parameters.arc_token.data()[self.Parameters.arc_token.default_key]

    @singleton_property
    def yt_token(self):
        return self.Parameters.yt_token.data()[self.Parameters.yt_token.default_key]

    def on_save(self):
        binary_task.LastRefreshableBinary.on_save(self)
        sdk2.Task.on_save(self)

    def on_enqueue(self):
        binary_task.LastRefreshableBinary.on_enqueue(self)
        sdk2.Task.on_enqueue(self)

    def on_wait(self, prev_status, status):
        binary_task.LastRefreshableBinary.on_wait(self, prev_status, status)
        sdk2.Task.on_wait(self, prev_status, status)

    def on_execute(self):
        self.download_and_extract_binaries()
        if self.Parameters.use_sources:
            self.patch_sources()
        self.prepare_environment()
        self.download_image()
        self.pre_run()
        self.run_program()
        self.post_run()

    def prepare_environment(self):
        # sudo service docker restart sometimes hangs up even with retries, so we execute task on another host by raising TemporaryError
        try:
            configure_docker(self)
        except RunCommandException:
            if self.Context.docker_retries < 10:
                self.Context.docker_retries += 1
                raise TemporaryError('Error while configuring docker')
            else:
                raise

        login_to_docker(self, self.Parameters.docker_login, self.Parameters.docker_token.data()[self.Parameters.docker_token.default_key])
        if self.Parameters.yt_token:
            initialize_yt_credentials(self.yt_token)
        if self.Parameters.aws_access_key_id and self.Parameters.aws_secret_access_key:
            initialize_aws_credentials(
                key_id=self.Parameters.aws_access_key_id.data()[self.Parameters.aws_access_key_id.default_key],
                secret_key=self.Parameters.aws_secret_access_key.data()[self.Parameters.aws_secret_access_key.default_key]
            )

    @singleton_property
    def binaries_branch_or_commit(self):
        return self.STABLE_COMMIT if self.Parameters.use_stable else self.Parameters.binaries_branch_or_commit

    @singleton_property
    def vcs_client(self):
        return VcsService(vcs_type=constants.VCS_ARC, arc_token=self.arc_token)

    def prepare_binaries(self, commit, branch=None):
        if not commit:
            raise Exception('commit is required')

        if binaries_are_ready(commit):
            return

        buildsdc_task = SdcBuildNoTests.find(
            status=Status.Group.EXECUTE + Status.Group.QUEUE + Status.Group.SUCCEED,
            hints=[commit],
            all_hints=True,
            children=True
        ).first()
        if buildsdc_task:
            self.set_info(
                'Waiting for task: <a href="{}">{} #{}</a> {}'.format(
                    get_task_link(buildsdc_task.id), sdk2.Task[buildsdc_task.id].type.name, buildsdc_task.id, sdk2.Task[buildsdc_task.id].Parameters.description
                ),
                do_escape=False
            )
            with self.memoize_stage.check_is_already_built_on_sandbox:
                raise sdk2.WaitTask(buildsdc_task.id, Status.Group.FINISH + Status.Group.BREAK)

        branch = branch or 'trunk'
        self.run_subtasks(
            [(
                SdcBuildNoTests,
                {
                    SdcBuildNoTests.Parameters.description: 'Prepare binaries for {} [{}]'.format(commit, branch),
                    SdcBuildNoTests.Parameters.commit: commit,
                    SdcBuildNoTests.Parameters.branch: branch,
                    SdcBuildNoTests.Parameters.vcs_type: 'arc',
                    SdcBuildNoTests.Parameters.vcs_steps_path: 'tc_build/build_sdc_no_tests',
                }
            )]
        )

    def download_and_extract_binaries(self):
        is_branch = not re.match(r'^[0-9a-f]{40}$', self.binaries_branch_or_commit)
        if not self.Context.commit or is_branch or self.Parameters.use_stable and self.Context.commit != self.binaries_branch_or_commit:
            if self.Parameters.recent_builded and is_branch:
                recent_commits = [c['commit'] for c in get_commits(self.vcs_client.arc_cli, self.binaries_branch_or_commit)]
                if not recent_commits:
                    raise Exception('No recent commits found for branch {}'.format(self.binaries_branch_or_commit))

                for commit in recent_commits:
                    logging.info('Checking binaries for commit %s', commit)
                    if binaries_are_ready(commit):
                        self.Context.commit = commit
                        break
                else:
                    raise Exception('Recent binaries not found on branch {}'.format(self.binaries_branch_or_commit))
            else:
                self.Context.commit = self.vcs_client.get_commit_from_branch_or_commit(self.binaries_branch_or_commit)
                self.Context.save()

        if not self.Parameters.recent_builded or not is_branch:
            self.prepare_binaries(self.Context.commit, self.binaries_branch_or_commit if is_branch else 'trunk')

        logging.info('Using binaries for commit %s', self.Context.commit)
        run_command(self, 'wget -c {} -O binaries.tar.gz'.format(get_binaries_url(self.Context.commit)), retries=3)
        self.sdc_root.mkdir(exist_ok=True, parents=True)
        run_command(self, 'tar -I pigz -xf binaries.tar.gz -C {}'.format(self.sdc_root))
        run_command(self, 'chmod -R 777 {}'.format(self.sdc_root))

    def patch_sources(self):
        with arc.sdc_checkout(
            self.arc_token, constants.ARC_MODE_MOUNT, self.Parameters.sources_branch,
            working_dir=self.path('arcadia').as_posix(),
            obj_store_path=self.path('store').as_posix()
        ) as arcadia_root:
            sdc_arc_root = Path(arcadia_root).joinpath(arc.ARC_SDC_REL_PATH)
            for file in sdc_arc_root.joinpath('simulator').rglob('*.py'):
                relative_path = file.relative_to(sdc_arc_root)
                binaries_file = self.sdc_root.joinpath('bazel-bin/sdc_install/lib/python3.6/dist-packages', relative_path)
                if not binaries_file.exists():
                    logging.debug('New file: %s', relative_path)
                    binaries_file.parent.mkdir(exist_ok=True, parents=True)
                    shutil.move(file.as_posix(), binaries_file.as_posix())
                elif not filecmp.cmp(binaries_file.as_posix(), file.as_posix()):
                    if '/proto/' not in file.as_posix():
                        logging.debug('Updating file: %s', relative_path)
                        shutil.move(file.as_posix(), binaries_file.as_posix())

    def download_image(self):
        run_command(self, './d --run ./sync', cwd=self.sdc_root.as_posix(), retries=3)

    @property
    def env_vars(self):
        return self.Parameters.env_vars

    @property
    def program(self):
        return self.Parameters.program

    def pre_run(self):
        pass

    def post_run(self):
        pass

    def on_cmd_error(self):
        pass

    def on_cmd_success(self):
        pass

    def run_program(self):
        env = {}
        env['YT_TOKEN'] = self.yt_token
        for k, v in self.env_vars.items():
            try:
                m = re.search(r'^(?P<secret_id>sec-\w+?)#(?P<secret_key>.*)$', v)
                if m:
                    v = sdk2.yav.Secret(m.group('secret_id')).data()[m.group('secret_key')]
            except TypeError:
                pass
            env[k] = v

        full_command = self.program
        logging.debug('Program: %s', full_command)
        if env:
            env_file = self.sdc_root.joinpath('env_vars')
            env_file.write_text(
                '\n'.join('export {}={}'.format(k, v) for k, v in sorted(env.items())),
                encoding='utf8'
            )
            full_command = 'source {}; {}'.format(env_file.name, full_command)
        # shell=True is important because of quotes escaping
        try:
            self.Context.started = True
            self.Context.save()
            run_command(
                self,
                './d --run "{}"'.format(full_command.replace('"', '\\"')), shell=True,
                cwd=self.sdc_root.as_posix(), logger='program',
                retries=self.Parameters.retries, print_logs=True
            )
        except RunCommandException:
            self.Context.failed = True
            self.Context.save()
            self.on_cmd_error()
            raise
        else:
            self.Context.failed = False
            self.Context.save()
            self.on_cmd_success()
        finally:
            if self.Parameters.program_output_path:
                program_output_path = self.sdc_root.joinpath(self.Parameters.program_output_path)
                if program_output_path.exists():
                    if not self.Parameters.output_as_parameter:
                        output_resource = OTHER_RESOURCE(self, 'program output', program_output_path)
                        sdk2.ResourceData(output_resource).ready()
                    else:
                        program_output = program_output_path.read_text(encoding='utf8')
                        if self.Parameters.output_is_json:
                            self.Parameters.json_output = json.loads(program_output)
                        else:
                            self.Parameters.output = program_output
                else:
                    logging.warning('Output path %s doesn\'t exist', program_output_path)
