# coding=utf-8
from __future__ import unicode_literals

from datetime import timedelta

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.common.types.notification import Transport
from sandbox.common.types.task import Status
from sandbox.projects.common import task_env
from sandbox.projects.metrika.utils.parameters import choices
from sandbox.projects.sdc.simulator.SdcSimulatorRunProgram import (
    SdcSimulatorRunProgram,
    CommonParameters,
)
from sandbox.projects.sdc.simulator.SdcSimulatorRunRegularity import REGULARITY_TAGS, resolve_commit_and_branch
from sandbox.projects.sdc.simulator.utils import get_simulator_api, SECRET_ID


class SdcSimulatorRegularityTasks(sdk2.Resource):
    restart_policy = ctr.RestartPolicy.IGNORE


STABLE_COMMIT = '83895f40d7b6cc2c41641ac839c47bff3f6783de'


class SdcSimulatorRunRegular(SdcSimulatorRunProgram):
    STABLE_COMMIT = STABLE_COMMIT

    class Context(SdcSimulatorRunProgram.Context):
        binaries_branch = None
        binaries_commit = None

    class Requirements(task_env.TinyRequirements):
        disk_space = 32 * 1024

    class Parameters(CommonParameters):
        use_stable = sdk2.parameters.Bool('Use stable program commit', default=True, description=STABLE_COMMIT)
        with use_stable.value[False]:
            binaries_branch_or_commit = sdk2.parameters.String('Binaries branch or commit hash', default='trunk')

        with sdk2.parameters.Group('Script parameters') as script_parameters:
            tag = sdk2.parameters.String('Tag', required=True, choices=choices(sorted(REGULARITY_TAGS)), default='test')
            frequency = sdk2.parameters.String('Frequency', choices=choices(['D1', 'H2', 'H3', 'H12', 'M30']))

            binaries_commit = sdk2.parameters.String('Binaries commit')
            binaries_for_metrics_commit = sdk2.parameters.String('Binaries for metrics commit')
            binaries_branch = sdk2.parameters.String('Binaries branch')

            simulator_api = sdk2.parameters.String('Simulator api')

            launch_id = sdk2.parameters.String('Launch Id', description='If you need to restart another launch task')

            ignore_day = sdk2.parameters.Bool('Ignore today\'s day of week')
            on_branch_update = sdk2.parameters.Bool('Run regularities only if the branch was updated')

            wait = sdk2.parameters.Bool('Wait regularity tasks')
            run_or_restart = sdk2.parameters.Bool('Run or restart', description='Run new or restart if previously task failed')

            regularity_retries = sdk2.parameters.Integer('Regularity retries count', default=0)

    @sdk2.header(title='State')
    def header(self):
        cells = [
            '<a href="{}/pool/regular?tag={}">Regularities</a>'.format(
                get_simulator_api(self.Parameters.simulator_api), self.Parameters.tag
            ),
        ]

        if self.Parameters.wait and self.Context.failed and self.status != Status.EXECUTING:
            cells.append(
                '<a href="https://sandbox.yandex-team.ru/tasks?tags={}&type=SDC_SIMULATOR_RUN_REGULARITY&status=BREAK">Failed Regularities tasks</a>'.format(self.launch_id),
            )
        elif self.Context.started:
            cells.append(
                '<a href="https://sandbox.yandex-team.ru/tasks?tags={}&type=SDC_SIMULATOR_RUN_REGULARITY">Regularities tasks</a>'.format(self.launch_id),
            )

        return '<table border="1" style="font-size: medium;"><tr>{}</tr></table>'.format(''.join('<td style="padding: 5px;">{}</td>'.format(cell) for cell in cells))

    def on_save(self):
        if not self.Parameters.binaries_commit and not self.Parameters.binaries_branch and not REGULARITY_TAGS.get(self.Parameters.tag):
            raise Exception('Specify binaries commit/branch in parameters')

        if self.Parameters.wait and not self.Parameters.notifications:
            self.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[Status.Group.BREAK],
                    recipients=['SDC-OFFLINE'],
                    transport=Transport.TELEGRAM,
                )
            ]

        SdcSimulatorRunProgram.on_save(self)

    @property
    def launch_id(self):
        return self.Parameters.launch_id or self.id

    @property
    def program(self):
        program = (
            'python3 -m simulator.pool.validate run-regular '
            '--tag {tag} --execution-type sandbox '
            '--launch-id {launch_id}'
        ).format(
            tag=self.Parameters.tag,
            launch_id=self.launch_id,
        )
        program += ' --binaries-commit {}'.format(self.Context.binaries_commit)
        if self.Parameters.frequency:
            program += ' --frequency {}'.format(self.Parameters.frequency)
        if self.Parameters.binaries_for_metrics_commit:
            program += ' --binaries-for-metrics-commit {}'.format(self.Parameters.binaries_for_metrics_commit)
        if self.Context.binaries_branch:
            program += ' --binaries-branch {}'.format(self.Context.binaries_branch)
        if self.Parameters.simulator_api:
            program += ' --api {}'.format(self.Parameters.simulator_api)
        if self.Parameters.run_or_restart:
            program += ' --run-or-restart'
        if self.Parameters.ignore_day:
            program += ' --ignore-day'
        if self.Parameters.on_branch_update:
            program += ' --on-branch-update'
        if self.Parameters.wait:
            program += ' --wait'
        if self.Parameters.regularity_retries:
            program += ' --regularity-retries {}'.format(self.Parameters.regularity_retries)
        return program

    @property
    def env_vars(self):
        return {
            'T__SECRET__SANDBOX_TOKEN': '{}#SANDBOX_TOKEN'.format(SECRET_ID),
        }

    def on_execute(self):
        resolve_commit_and_branch(self)
        SdcSimulatorRunProgram.on_execute(self)

    def on_cmd_error(self):
        self.Parameters.description = '[{}] <a href="https://sandbox.yandex-team.ru/tasks?tags={}&type=SDC_SIMULATOR_RUN_REGULARITY&status=BREAK">Failed regularities on {}</a>'.format(
            (self.created + timedelta(hours=3)).date(),
            self.launch_id, self.Parameters.tag
        )
