# coding=utf-8
from __future__ import unicode_literals

import json
import logging
from datetime import datetime, timedelta

import requests

from sandbox import sdk2
from sandbox.common.patterns import singleton_property
from sandbox.common.types.task import Status
from sandbox.projects.abc.client import AbcClient
from sandbox.projects.common.decorators import retries
from sandbox.projects.metrika.utils.parameters import choices
from sandbox.projects.sdc.simulator.SdcSimulatorRunProgram import (
    SdcSimulatorRunProgram,
    CommonParameters,
)
from sandbox.projects.sdc.simulator.utils import (
    get_simulator_api, get_ic_api, is_branch, SECRET_ID, get_simulator_api_from_ic_api,
)

STABLE_COMMIT = 'b8808497d1d770fc5703733402ba4e57367ac171'

REGULARITY_TAGS = {
    'master': 'rel_mgr.current_master',
    'master_lts': 'rel_mgr.current_master_lts',
    'midnight_dev': 'tags/groups/sdg/sdc/midnight-dev',
    'release_candidate': 'rel_mgr.current_release',
    'prestable': 'rel_mgr.current_prestable',
    'robot_rc': 'rel_mgr.robot_current_release',
    'robot_master': 'rel_mgr.robot_current_master',
    'mesh_qa_tests': 'tags/groups/sdg/sdc/robot-bl/mesh-qa-tests',
    'mesh_master': 'tags/groups/sdg/sdc/robot-bl/mesh-master',
    'trunk': 'trunk',
    'test': 'tags/groups/sdg/sdc/midnight-dev',
}


@retries(10)
def get_branch_from_tag(tag):
    branch = REGULARITY_TAGS[tag]
    if branch.startswith('rel_mgr.'):
        resp = requests.get('https://ci.sdc.yandex-team.ru/api/eventbus/parameter?name={}'.format(branch))
        resp.raise_for_status()
        try:
            return json.loads(resp.json()[0]['value'])['branch']
        except Exception:
            msg = 'Error while resolving branch from tag'
            logging.exception(msg)
            raise Exception(msg)
    else:
        return branch


def resolve_commit_and_branch(task):
    with task.memoize_stage.resolve_branch(commit_on_entrance=False):
        if not task.Parameters.binaries_branch:
            if REGULARITY_TAGS.get(task.Parameters.tag):
                binaries_branch = get_branch_from_tag(task.Parameters.tag)
            elif is_branch(task.Parameters.binaries_commit):
                binaries_branch = task.Parameters.binaries_commit
            else:
                binaries_branch = None
        else:
            binaries_branch = task.Parameters.binaries_branch

        if not task.Parameters.binaries_commit:
            binaries_commit = task.vcs_client.get_commit_from_branch_or_commit(binaries_branch)
        else:
            binaries_commit = task.Parameters.binaries_commit

        task.Context.binaries_branch = binaries_branch
        task.Context.binaries_commit = binaries_commit
        task.Context.save()


class SdcSimulatorRunRegularity(SdcSimulatorRunProgram):
    STABLE_COMMIT = STABLE_COMMIT

    class Context(SdcSimulatorRunProgram.Context):
        binaries_branch = None
        binaries_commit = None
        binaries_for_metrics_commit = None
        pulsar_datetime = None

    class Requirements(SdcSimulatorRunProgram.Requirements):
        environments = [
            sdk2.environments.PipEnvironment('yandex-yt'),
        ]

    class Parameters(CommonParameters):
        kill_timeout = 36 * 60 * 60

        use_stable = sdk2.parameters.Bool('Use stable program commit', default=True, description=STABLE_COMMIT)
        with use_stable.value[False]:
            binaries_branch_or_commit = sdk2.parameters.String('Binaries branch or commit hash')

        with sdk2.parameters.Group('Script parameters') as script_parameters:
            process_id = sdk2.parameters.Integer('Process', required=True)
            regularity_id = sdk2.parameters.Integer('Regularity', required=True)
            tag = sdk2.parameters.String('Tag', required=True, choices=choices(sorted(REGULARITY_TAGS)), default='test')
            version = sdk2.parameters.String('Version', required=True, choices=choices(['latest', 'latest_stable']), default='latest')

            binaries_commit = sdk2.parameters.String('Binaries commit')
            binaries_for_metrics_commit = sdk2.parameters.String('Binaries for metrics commit')
            binaries_branch = sdk2.parameters.String('Binaries branch')

            simulator_api = sdk2.parameters.String('Simulator api')
            ic_api = sdk2.parameters.String('IC api')
            ov_api = sdk2.parameters.String('OV api')

            parent_id = sdk2.parameters.String('Parent Id', description='If you need to restart another regularity task')

            run_or_restart = sdk2.parameters.Bool('Run or restart', description='Run new or restart if previously task failed')

            abc_token = sdk2.parameters.YavSecret('ABC oauth token', default='{}#ABC_TOKEN'.format(SECRET_ID))

    @sdk2.header(title='State')
    def header(self):
        if self.Parameters.version == 'latest':
            process_query = ''
        elif self.Parameters.version == 'latest_stable':
            process_query = '?stable=1'
        else:
            process_query = '?version={}'.format(self.Parameters.version)

        cells = [
            '<a href="{simapi}/pool/process/{process_id}{process_query}">Process {process_id}</a>'.format(
                simapi=get_simulator_api(self.Parameters.simulator_api),
                process_id=self.Parameters.process_id, process_query=process_query
            ),
            '<a href="{simapi}/pool/process/{process_id}/regular/{regularity_id}">Regularity {regularity_id}'.format(
                simapi=get_simulator_api(self.Parameters.simulator_api),
                process_id=self.Parameters.process_id, regularity_id=self.Parameters.regularity_id,
            ),
        ]

        if self.Context.failed and self.status != Status.EXECUTING:
            cells.append('<a href="{}/task?mine=false&task_type=PoolRecipeTask&parent_id={}&state=FAILED">IC failed tasks</a>'.format(
                get_simulator_api_from_ic_api(get_ic_api(self.Parameters.ic_api)), self.parent_id
            ))
        elif self.Context.started:
            cells.append('<a href="{}/task?mine=false&task_type=PoolRecipeTask&parent_id={}">IC tasks</a>'.format(
                get_simulator_api_from_ic_api(get_ic_api(self.Parameters.ic_api)), self.parent_id
            ))

        if self.Context.st_key:
            cells.append('<a href="https://st.yandex-team.ru/{0}">{0}</a>'.format(self.Context.st_key))

        return '<table border="1" style="font-size: medium;"><tr>{}</tr></table>'.format(''.join('<td style="padding: 5px;">{}</td>'.format(cell) for cell in cells))

    @singleton_property
    @retries(3)
    def regularity(self):
        r = requests.get(
            '{}/api/pool/process/{}/regularity/{}'.format(
                get_simulator_api(self.Parameters.simulator_api),
                self.Parameters.process_id, self.Parameters.regularity_id
            )
        )
        r.raise_for_status()
        return r.json()['item']

    def on_save(self):
        if not self.Parameters.binaries_commit and not self.Parameters.binaries_branch and not REGULARITY_TAGS.get(self.Parameters.tag):
            raise Exception('Specify binaries commit/branch in parameters')

        description = '[{}] process <b>{}@{}</b>, regularity <b>{}@{}</b>'.format(
            (self.created + timedelta(hours=3)).date(),
            self.Parameters.process_id, self.Parameters.version,
            self.Parameters.regularity_id, self.Parameters.tag,
        )

        self.Parameters.description = description

        SdcSimulatorRunProgram.on_save(self)

    @property
    def parent_id(self):
        return self.Parameters.parent_id or 'sb{}'.format(self.id)

    @property
    def program(self):
        program = (
            'python3 -m simulator.pool.validate run-regularity '
            '--execution-type isolate_cloud --process-id {process_id} '
            '--regularity-id {regularity_id} --tag {tag} '
            '--version {version} '
            '--binaries-commit {binaries_commit} '
            '--parent-id {parent_id} --wait '
            '--pulsar-datetime "{pulsar_datetime}" '
        ).format(
            process_id=self.Parameters.process_id,
            regularity_id=self.Parameters.regularity_id,
            tag=self.Parameters.tag,
            version=self.Parameters.version,
            binaries_commit=self.Context.binaries_commit,
            parent_id=self.parent_id,
            pulsar_datetime=self.Context.pulsar_datetime,
        )
        if self.Parameters.binaries_for_metrics_commit:
            program += ' --binaries-for-metrics-commit {}'.format(self.Parameters.binaries_for_metrics_commit)
        if self.Context.binaries_branch:
            program += ' --binaries-branch {}'.format(self.Context.binaries_branch)
        if self.Parameters.simulator_api:
            program += ' --api {}'.format(self.Parameters.simulator_api)
        if self.Parameters.run_or_restart:
            program += ' --run-or-restart'
        return program

    @property
    def env_vars(self):
        env_vars = {
            'PULSAR_TOKEN': '{}#PULSAR_TOKEN'.format(SECRET_ID),
            'T__SECRET__ISOLATE_CLOUD_TOKEN': '{}#ISOLATE_CLOUD_TOKEN'.format(SECRET_ID),
            'YAV_TOKEN': '{}#YAV_TOKEN'.format(SECRET_ID),
            'ARC_TOKEN': '{}#ARC_TOKEN'.format(SECRET_ID),
            'DATALENS_TOKEN': '{}#DATALENS_TOKEN'.format(SECRET_ID),
            'ABC_TOKEN': '{}#ABC_TOKEN'.format(SECRET_ID),
            'ST_TOKEN': '{}#ST_TOKEN'.format(SECRET_ID),
            'T__SECRET__SANDBOX_TOKEN': '{}#SANDBOX_TOKEN'.format(SECRET_ID),
        }
        if self.Parameters.ic_api:
            env_vars['T__ISOLATE_CLOUD_SERVER_URL'] = self.Parameters.ic_api
        if self.Parameters.ov_api:
            env_vars['OV_API_HOST'] = self.Parameters.ov_api
        return env_vars

    @retries(10)
    def get_nda_link(self, link):
        return requests.get('https://nda.ya.ru/--?url={}'.format(link)).text

    def on_cmd_error(self):
        import yt.wrapper as yt

        yt_client = yt.YtClient(token=self.yt_token, proxy='hahn')
        root = '//home/selfdriving/simulator/pools/regular/{}/{}/{}/{}'.format(
            self.Parameters.regularity_id,
            self.Parameters.tag,
            self.Parameters.version,
            self.parent_id
        )
        try:
            self.Context.st_key = yt_client.get_attribute(root, 'st_key')
        except Exception:
            logging.exception('Error while getting ST key')

        if self.Context.st_key and self.Context.st_key not in self.Parameters.description:
            try:
                self.Parameters.description = '{} <a href="{}">{}</a>'.format(
                    self.Parameters.description,
                    self.get_nda_link('https://st.yandex-team.ru/{}'.format(self.Context.st_key)),
                    self.Context.st_key
                )
            except Exception:
                logging.exception('Error while getting NDA link')

    def resolve_binaries_for_metrics_commit(self):
        with self.memoize_stage.resolve_binaries_for_metrics_commit(commit_on_entrance=False):
            if self.Parameters.binaries_for_metrics_commit:
                binaries_for_metrics_commit = self.Parameters.binaries_for_metrics_commit
            else:
                binaries_for_metrics_commit = self.regularity['config']['binariesForMetricsCommit']

            if binaries_for_metrics_commit:
                self.Context.binaries_for_metrics_commit = self.vcs_client.get_commit_from_branch_or_commit(
                    binaries_for_metrics_commit
                )
                self.Context.save()

    @retries(3)
    def get_duty(self, slug):
        abc_client = AbcClient(self.Parameters.abc_token.data()[self.Parameters.abc_token.default_key])
        return abc_client.get_current_duty_login(schedule_slug=slug)

    def on_execute(self):
        if not self.Context.pulsar_datetime:
            self.Context.pulsar_datetime = datetime.now().strftime('%Y-%m-%d %H:%M:%S')

        with self.memoize_stage.assignee(commit_on_entrance=False):
            try:
                assignee = self.regularity['config']['startrek']['assignee']
                if assignee:
                    if assignee.startswith('@'):
                        assignee = self.get_duty(assignee[1:])

                    if assignee and assignee not in self.Parameters.tags:
                        self.Parameters.tags = self.Parameters.tags + [assignee]
            except KeyError:
                pass

        resolve_commit_and_branch(self)
        self.prepare_binaries(self.Context.binaries_commit, self.Context.binaries_branch)

        self.resolve_binaries_for_metrics_commit()
        if self.Context.binaries_for_metrics_commit:
            self.prepare_binaries(self.Context.binaries_for_metrics_commit)

        SdcSimulatorRunProgram.on_execute(self)
