# coding=utf-8
from __future__ import unicode_literals

import logging

from sandbox import sdk2
from sandbox.common.patterns import singleton_property
from sandbox.common.types.task import Status
from sandbox.projects.common import binary_task
from sandbox.projects.common import task_env
from sandbox.projects.common.decorators import retries
from sandbox.projects.common.reactor.reactor_api import ReactorApi
from sandbox.projects.metrika.utils.parameters import choices
from sandbox.projects.sdc.simulator.SdcSimulatorRunRegularity import (
    REGULARITY_TAGS,
    get_branch_from_tag,
)
from sandbox.projects.sdc.simulator.utils import SECRET_ID


class SdcSimulatorTriggerRegularReaction(binary_task.LastRefreshableBinary, sdk2.Task):
    class Context(sdk2.Context):
        processed_tags = []

    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(sdk2.Parameters):
        updatable_tags = sdk2.parameters.Bool('Updatable tags')
        frequency = sdk2.parameters.String('Frequency', choices=choices(['D1', 'H2', 'H3', 'H12', 'M30']))

        with sdk2.parameters.Output(reset_on_restart=True):
            tags_branches = sdk2.parameters.Dict('Tag to branch mapping')

        reactor_token = sdk2.parameters.YavSecret('Reactor oauth token', default='{}#NIRVANA_TOKEN'.format(SECRET_ID))

        _binary = binary_task.binary_release_parameters_list(none=True)

    def on_save(self):
        binary_task.LastRefreshableBinary.on_save(self)
        sdk2.Task.on_save(self)

    def on_enqueue(self):
        binary_task.LastRefreshableBinary.on_enqueue(self)
        sdk2.Task.on_enqueue(self)

    def on_wait(self, prev_status, status):
        binary_task.LastRefreshableBinary.on_wait(self, prev_status, status)
        sdk2.Task.on_wait(self, prev_status, status)

    @singleton_property
    def reactor_token(self):
        return self.Parameters.reactor_token.data()[self.Parameters.reactor_token.default_key]

    @singleton_property
    def reactor_client(self):
        return ReactorApi(self.reactor_token)

    @retries(5, max_delay=5)
    def create_artifact_instance(self, tag, on_branch_update):
        artifact_id = self.reactor_client.instantiateArtifact(
            '/selfdriving/simulator/offline/regular_trigger',
            metadata={
                '@type': '/yandex.reactor.artifact.StringDictArtifactValueProto',
                'keyValue': {
                    'tag': tag,
                    'on_branch_update': on_branch_update,
                    'task': self.id,
                    'frequency': self.Parameters.frequency,
                }
            }
        )['artifactInstanceId']
        logging.debug('Created artifact: %s', artifact_id)

    @singleton_property
    def previous_launch_mapping(self):
        previous_task = sdk2.Task.find(task_type=self.type, status=Status.Group.SUCCEED, input_parameters={
            self.type.Parameters.updatable_tags.name: True,
        }).limit(1).first()
        if previous_task:
            return previous_task.Parameters.tags_branches
        else:
            logging.debug('Previous launch not found')
            return {}

    @singleton_property
    def current_launch_mapping(self):
        tags_branches = {}
        for tag, param in REGULARITY_TAGS.items():
            if param.startswith('rel_mgr'):
                tags_branches[tag] = get_branch_from_tag(tag)

        self.Parameters.tags_branches = tags_branches
        return tags_branches

    def is_tag_updated(self, tag):
        previous_branch = self.previous_launch_mapping.get(tag)

        if not previous_branch:
            return False

        current_branch = self.current_launch_mapping[tag]
        logging.debug('Previous branch was: %s', previous_branch)
        logging.debug('Current branch is: %s', current_branch)
        return current_branch != previous_branch

    def on_execute(self):
        self.Parameters.description = ''
        on_branch_update = False
        if not self.Parameters.updatable_tags:
            tags = set(REGULARITY_TAGS.keys()) - {'test'}
        else:
            tags = [tag for tag, branch in REGULARITY_TAGS.items() if branch.startswith('rel_mgr')]
            on_branch_update = True
            logging.debug('Previous branches: %s', self.previous_launch_mapping)
            logging.debug('Current branches: %s', self.current_launch_mapping)

        error_tags = []
        for tag in sorted(tags):
            logging.debug('Processing tag: %s', tag)
            if tag in self.Context.processed_tags:
                logging.debug('Tag %s has been already processed', tag)
                continue

            if self.Parameters.updatable_tags:
                if not self.is_tag_updated(tag):
                    logging.debug('Tag %s hasn\'t been updated', tag)
                    continue

            try:
                self.create_artifact_instance(tag, on_branch_update=on_branch_update)
            except Exception:
                logging.exception('Error while creating artifact instance')
                error_tags.append(tag)
            else:
                self.Context.processed_tags.append(tag)

        if error_tags:
            self.Parameters.description = 'Error while creating artifact instance for tags {}'.format(error_tags)
            raise Exception(self.Parameters.description)
