# coding=utf-8

from sandbox import sdk2
import sandbox.common.types.resource
import datetime
import logging
import os


class BackupBigrtStates(sdk2.Task):
    class Requirements(sdk2.Requirements):
        # configure this for your task, the more accurate - the better
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        description = "Backup bigRt states"
        yt_token = sdk2.parameters.YavSecret("YtToken", description="Yt token name", required=True, default="sec-01e4v4qf7gjp639tq577m42wgx")

    def on_save(self):
        self.Requirements.tasks_resource = (
            sdk2.service_resources.SandboxTasksBinary.find(
                attrs={"target": "search_pers/BackupBigrtStates"}
            ).first()
        )
        logging.info("task binary: {}".format(self.Requirements.tasks_resource))

    def on_execute(self):
        super(BackupBigrtStates, self).on_execute()
        logging.basicConfig(level=logging.DEBUG, format="%(asctime)s %(levelname)s [%(threadName)s] %(name)s:%(message)s")
        logging.info("checking that tasks resource is properly set")
        logging.info("current tasks_resource: {}".format(
            self.Requirements.tasks_resource))
        if self.Requirements.tasks_resource is None:
            raise sandbox.common.errors.TaskFailure(
                "self.Requirements.tasks_resource is not set for BackupBigrtStates")

        for key in self.Parameters.yt_token.data():
            logging.info("key {} in yt_token".format(key))

        assert self.Parameters.yt_token.value(), "yt_token value is empty"

        ytToken = self.Parameters.yt_token.value()
        tmToken = self.Parameters.yt_token.value()

        os.environ["YT_TOKEN"] = ytToken

        from quality.personalization.big_rt.backup.lib.data_types import PathWithCluster
        from quality.personalization.big_rt.backup.lib.locate_async_replica import locate_async_replica
        from quality.personalization.big_rt.backup.lib.logarithmic_backup import cleanup, BackupSettings
        from quality.personalization.big_rt.backup.lib.transfer import dyntable_transfer
        from collections import namedtuple
        import concurrent.futures
        from concurrent.futures import ThreadPoolExecutor

        BackupRecord = namedtuple("BackupRecord", ["SrcCluster", "SrcTable", "SrcTmpPrefix", "DstCluster", "DstPrefix"])

        def process_backup_record(rec, ytToken, tmToken):
            async_replica = locate_async_replica(PathWithCluster(rec.SrcCluster, rec.SrcTable), ytToken=ytToken)
            iso8601 = datetime.datetime.now().astimezone().replace(microsecond=0).isoformat()
            dyntable_transfer(
                src=async_replica,
                dst=PathWithCluster(rec.DstCluster, rec.DstPrefix + "/" + iso8601),
                tmpPrefix=rec.SrcTmpPrefix,
                ytToken=ytToken,
                tmToken=tmToken
            )

            cleanup(PathWithCluster(rec.DstCluster, rec.DstPrefix), BackupSettings(6, datetime.timedelta(hours=1), 2.0), ytToken=ytToken)

        backupRecords = [
            BackupRecord('markov', '//home/searchpers/production/States', '//home/searchpers/backup/', 'hahn', '//home/searchpers/backup/UserHistory'),
            BackupRecord('markov', '//home/searchpers/prestable_A/user_history/States', '//home/searchpers/backup/', 'hahn', '//home/searchpers/backup/UserHistory_prestable_A'),
            BackupRecord('markov', '//home/searchpers/prestable_B/user_history/States', '//home/searchpers/backup/', 'hahn', '//home/searchpers/backup/UserHistory_prestable_B'),
            BackupRecord('markov', '//home/searchpers/prestable_C/user_history/States', '//home/searchpers/backup/', 'hahn', '//home/searchpers/backup/UserHistory_prestable_C'),
            BackupRecord('markov', '//home/searchpers/search_middle/mega_aggregator/production/States', '//home/searchpers/backup/', 'hahn', '//home/searchpers/backup/AggregatorMiddle'),
            BackupRecord('markov', '//home/searchpers/spylog/aggregate/production/States', '//home/searchpers/backup/', 'hahn', '//home/searchpers/backup/Spylog'),
            BackupRecord('markov', '//home/searchpers/turbo/aggregate/production/States', '//home/searchpers/backup/', 'hahn', '//home/searchpers/backup/Turbo'),
        ]

        failedTasksCount = 0
        with ThreadPoolExecutor(max_workers=2) as e:
            backupTasks = list()
            for rec in backupRecords:
                backupTasks.append(e.submit(process_backup_record, rec, ytToken, tmToken))

            for rec, future in zip(backupRecords, concurrent.futures.as_completed(backupTasks)):
                try:
                    future.result()
                except Exception as exc:
                    logging.error("Backup task {} finished with exception {}".format(repr(rec), exc))
                    failedTasksCount = failedTasksCount + 1

        if failedTasksCount:
            raise RuntimeError("Backup tasks failed {}/{}. See logs for details".format(failedTasksCount, len(backupRecords)))
