# -*- coding: utf-8 -*-

from sandbox.common import errors
from sandbox import sdk2
import sandbox.projects.common.arcadia.sdk as arcadiasdk
import sandbox.projects.common.constants as constants
import sandbox.projects.search_pers.resources as search_pers_resources

import logging
import os
import shutil


MONITORING_TOOLS_ARCADIA_RELATIVE_PATH = os.path.join('quality', 'pers', 'monitoring_tools')


class BuildSearchPersMonitoringTools(sdk2.Task):
    """Build archive with binaries of search personalization monitoring tools"""

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl('Path to Arcadia svn')
        archive_ttl = sdk2.parameters.Integer('Archive ttl')

    def on_execute(self):
        build_directory_path = str(self.path('build'))

        arcadia_path = sdk2.svn.Arcadia.get_arcadia_src_dir(self.Parameters.arcadia_url)
        arcadia_revision = sdk2.svn.Arcadia.get_revision(arcadia_path)
        logging.info('Building tools at revision {}'.format(arcadia_revision))

        arcadiasdk.do_build(
            constants.YMAKE_BUILD_SYSTEM,
            arcadia_path,
            [MONITORING_TOOLS_ARCADIA_RELATIVE_PATH],
            results_dir=build_directory_path,
            clear_build=True
        )

        monitoring_tools_resource = search_pers_resources.SearchPersMonitoringTools(
            self,
            'Search personalization monitoring tools archive',
            'search_pers_monitoring_tools',
            ttl=self.Parameters.archive_ttl,
            revision=arcadia_revision
        )
        monitoring_tools_resource_data = sdk2.ResourceData(monitoring_tools_resource)
        monitoring_tools_resource_data_path = str(monitoring_tools_resource_data.path)
        os.makedirs(monitoring_tools_resource_data_path)

        binaries_path = os.path.join(build_directory_path, MONITORING_TOOLS_ARCADIA_RELATIVE_PATH)
        for tool_name in os.listdir(binaries_path):
            logging.info('Copying tool {}'.format(tool_name))
            tool_directory_path = os.path.join(binaries_path, tool_name)
            if not os.path.isdir(tool_directory_path):
                raise errors.TaskFailure(
                    'Result of building {} tool is not directory'.format(tool_name)
                )
            tool_directory_file_names = os.listdir(tool_directory_path)
            if len(tool_directory_file_names) != 1:
                raise errors.TaskFailure(
                    'Expected to have exactly one binary for tool {}'.format(tool_name)
                )
            src_tool_binary_file_path = os.path.join(
                tool_directory_path,
                tool_directory_file_names[0]
            )
            dst_tool_binary_file_path = os.path.join(
                monitoring_tools_resource_data_path,
                tool_directory_file_names[0]
            )
            shutil.copyfile(src_tool_binary_file_path, dst_tool_binary_file_path)
            os.chmod(dst_tool_binary_file_path, 0o754)

        monitoring_tools_resource_data.ready()
