# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as subprocess
import sandbox.projects.search_pers.resources as search_pers_resources
import sandbox.projects.search_pers.common.rtmr as search_pers_rtmr
import sandbox.common.types.resource as ctr

import codecs
import json
import os
import requests
import time


class CalcSearchPersProfileSessionsRtmrFreshStats(sdk2.Task):
    """Calculate and upload to Solomon profile sessions Rtmr fresh statistics for active users"""

    class Parameters(search_pers_rtmr.WithRtmrRequestParameters):
        with sdk2.parameters.Group('Calculate statistics parameters') as calc_stats_params:
            monitoring_tools_resource = sdk2.parameters.Resource(
                'User profiles monitoring tools resource',
                resource_type=search_pers_resources.SearchPersMonitoringTools,
                state=ctr.State.READY,
                required=True,
            )
            users_count_limit = sdk2.parameters.Integer(
                'Maximum number of active users',
                default_value=100,
                required=True,
            )
            user_sessions_records_count_limit = sdk2.parameters.Integer(
                'Maximum number of active user sessions records',
                default_value=10000,
                required=True,
            )
            profile_sessions_stats_delays = sdk2.parameters.List(
                'Maximum time distance (in seconds) '
                'from sessions to profile for calculating statistics',
                value_type=sdk2.parameters.Integer,
                default=[60, 120, 180],
                required=True,
            )
            with sdk2.parameters.String('Calculate statistics for service', multiline=True) as\
                    service:
                service.values.web = service.Value(default=True)
                service.values.video = None

    def get_main_service_spec(self):
        service_to_main_service_spec = {
            'web': 'web',
            'video': 'vid'
        }
        return service_to_main_service_spec[self.Parameters.service]

    def get_user_profiles_table_name(self):
        service_to_user_profiles_table_name = {
            'web': 'user_profiles/19700101',
            'video': 'user_profiles/19700101/vid'
        }
        return service_to_user_profiles_table_name[self.Parameters.service]

    def get_user_profiles_solomon_service_name(self):
        return '{}_user_profiles'.format(self.Parameters.service)

    def configure_monitoring_tool(self, monitoring_tool_name):
        resource_data = sdk2.ResourceData(self.Parameters.monitoring_tools_resource)
        monitoring_tool_file_path = os.path.join(str(resource_data.path), monitoring_tool_name)
        return monitoring_tool_file_path

    def configure_rtmr_fetcher(self):
        return search_pers_rtmr.RtmrFetcher(
            self.configure_monitoring_tool('rtmr_fetch'),
            self.Parameters
        )

    def configure_profile_sessions_stats_tool(self):
        return self.configure_monitoring_tool('profile_sessions_stats')

    def allocate_resource(self, resource, content):
        resource_data = sdk2.ResourceData(resource)
        resource_data_file_path = str(resource_data.path)
        with open(resource_data_file_path, 'w') as resource_data_file:
            resource_data_file.write(content)
        resource_data.ready()
        return resource_data_file_path

    def allocate_user_ids_resource(self, content):
        user_ids_resource = search_pers_resources.SearchPersUserIds(
            self,
            'Search personalization active user ids for Rtmr fresh profile sessions statistics',
            'active_user_ids',
            ttl=2
        )
        return self.allocate_resource(user_ids_resource, content)

    def fetch_rtmr_active_user_ids(self, rtmr_fetcher):
        with sdk2.helpers.ProcessLog(self, logger='fetch_rtmr_active_user_ids') as logger:
            rtmr_output = rtmr_fetcher.fetch_keys(
                keys=['user_actions_json'],
                table='active_keys',
                stderr=logger.stderr
            )
        keys = map(lambda x: x[2], rtmr_output)
        user_ids = map(lambda x: x if x.startswith('uu/') else 'y' + x, keys)
        user_ids = list(set(user_ids))
        if self.Parameters.users_count_limit < len(user_ids):
            user_ids = user_ids[:self.Parameters.users_count_limit]
        return self.allocate_user_ids_resource('\n'.join(user_ids) + '\n')

    def allocate_user_sessions_resource(self, content):
        user_sessions_resource = search_pers_resources.SearchPersUserSessions(
            self,
            'Search personalization active user sessions for Rtmr fresh profile sessions statistics',
            'active_user_sessions',
            ttl=2
        )
        return self.allocate_resource(user_sessions_resource, content)

    def fetch_rtmr_user_sessions(self, rtmr_fetcher, user_ids_file_path):
        with sdk2.helpers.ProcessLog(self, logger='fetch_rtmr_user_sessions') as logger:
            rtmr_staff_output = rtmr_fetcher.fetch_keys_file(
                user_ids_file_path,
                'user_sessions/19700101/raw/yandex_staff/main_service/{}/postproc'.format(
                    self.get_main_service_spec()
                ),
                logger.stderr
            )
            rtmr_output = rtmr_fetcher.fetch_keys_file(
                user_ids_file_path,
                'user_sessions/19700101/raw/main_service/{}/postproc'.format(
                    self.get_main_service_spec()
                ),
                logger.stderr
            )
        user_sessions = map(lambda x: '\t'.join(x), rtmr_staff_output)
        user_sessions += map(lambda x: '\t'.join(x), rtmr_output)
        if self.Parameters.user_sessions_records_count_limit < len(user_sessions):
            user_sessions = user_sessions[:self.Parameters.user_sessions_records_count_limit]
        user_sessions_repr = codecs.encode('\n'.join(user_sessions) + '\n', 'utf-8')
        return self.allocate_user_sessions_resource(user_sessions_repr)

    def allocate_user_profiles_resource(self, content):
        user_profiles_resource = search_pers_resources.SearchPersUserProfiles(
            self,
            'Search personalization active user profiles for Rtmr fresh profile session statistics',
            'active_user_profiles',
            ttl=2,
        )
        return self.allocate_resource(user_profiles_resource, content)

    def fetch_rtmr_user_profiles(self, rtmr_fetcher, user_ids_file_path):
        request_timestamp = time.time()
        with sdk2.helpers.ProcessLog(self, logger='fetch_rtmr_user_profiles') as logger:
            rtmr_output = rtmr_fetcher.fetch_keys_file(
                user_ids_file_path, self.get_user_profiles_table_name(), logger.stderr
            )
        user_profiles = map(
            lambda x: '{}\t{}\t{}'.format(x[0], int(request_timestamp), codecs.encode(x[2], 'utf-8')),
            rtmr_output,
        )
        return self.allocate_user_profiles_resource('\n'.join(user_profiles) + '\n')

    def calculate_profile_sessions_statistics(
            self, user_sessions_file_path, user_profiles_file_path):
        profile_sessions_stats_tool_file_path = self.configure_profile_sessions_stats_tool()
        profile_sessions_stats_tool_command = [
            profile_sessions_stats_tool_file_path,
            '--sessions', user_sessions_file_path,
            '--profiles', user_profiles_file_path,
            '--service', self.get_main_service_spec()
        ]
        for delay in self.Parameters.profile_sessions_stats_delays:
            profile_sessions_stats_tool_command += ['--delay', str(delay)]
        with sdk2.helpers.ProcessLog(self, logger='calculate_profile_sessions_statistics') as logger:
            return json.loads(
                subprocess.check_output(profile_sessions_stats_tool_command, stderr=logger.stderr)
            )

    def upload_statistics_to_solomon(self, statistics):
        now = int(time.time())
        data = {
            'commonLabels': {
                'project': 'search_pers',
                'cluster': 'rtmr',
                'service': self.get_user_profiles_solomon_service_name()
            },
            'sensors': [
                {
                    'labels': {'sensor': key},
                    'ts': now,
                    'value': float(statistics[key])
                } for key in statistics.keys()
            ]
        }
        headers = {
            'Content-type': 'application/json',
        }
        response = requests.post(
            "http://api.solomon.search.yandex.net/push/json",
            data=json.dumps(data),
            headers=headers,
            timeout=15,
        )
        response.raise_for_status()

    def on_execute(self):
        rtmr_fetcher = self.configure_rtmr_fetcher()
        user_ids_file_path = self.fetch_rtmr_active_user_ids(rtmr_fetcher)
        user_sessions_file_path = self.fetch_rtmr_user_sessions(rtmr_fetcher, user_ids_file_path)
        user_profiles_file_path = self.fetch_rtmr_user_profiles(rtmr_fetcher, user_ids_file_path)
        self.upload_statistics_to_solomon(
            self.calculate_profile_sessions_statistics(
                user_sessions_file_path, user_profiles_file_path
            )
        )
