import json
import logging
import os
import pipes
import re
from collections import OrderedDict
from datetime import datetime

import jinja2
import numpy as np
import requests

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.projects.search_velocity.LightPageReviser.container import LightPageReviserContainer
from sandbox.projects.search_velocity.LightPageReviser.report import LightPageReviserReport
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.process import run_process

logger = logging.getLogger(__name__)

PERCENTILES = (25, 50, 75, 95, 99)

AUDITS = (
    'first-meaningful-paint',
    'speed-index',
    'time-to-first-byte',
    'interactive',
    'total-byte-weight'
)

METRICS = AUDITS + ('score',)

YT_TOKEN_NAME = 'robot-drunken-flash-yt'
YT_TOKEN_OWNER = 'SANDBOX_CI_SEARCH_INTERFACES'
YT_PATH = '//home/trencher/lighthouse'

RETRY_ON_FAIL_ATTEMPTS = 20


class LightPageReviserBase(sdk2.Task):
    """
    Run lighthouse test
    """

    class Requirements(sdk2.Requirements):
        disk_space = 10000
        dns = ctm.DnsType.DNS64
        ram = 4096

        environments = [
            PipEnvironment('yandex-yt')
        ]

    class Parameters(sdk2.Parameters):
        _container = LightPageReviserContainer(
            'LXC container with lighthouse and browsers',
            required=True
        )

        with sdk2.parameters.Group('Reviser params') as urls_params:
            urls = sdk2.parameters.String('URLs')
            iterations = sdk2.parameters.Integer('Iterations', default=10)
            perf_only = sdk2.parameters.Bool('Performance audits only', default=True)

            with sdk2.parameters.String('Platform') as platform:
                platform.values['desktop'] = platform.Value('desktop', default=True)
                platform.values['touch'] = platform.Value('touch')

        with sdk2.parameters.Group('Authorization params') as auth_params:
            need_auth = sdk2.parameters.Bool(
                'Need authorization',
                description='required for staff, tracker, etc \n https://passport.yandex-team.ru/auth',
                sub_fields={'true': ['need_auth_login', 'need_auth_password', 'need_auth_owner']},
                default_value=False,
            )

            need_auth_login = sdk2.parameters.String(
                'Login (Sandbox Vault Name)',
                description='See https://wiki.yandex-team.ru/Sandbox/vault/#web',
                required=True,
                default='light_page_reviser_login',
            )

            need_auth_password = sdk2.parameters.String(
                'Password (Sandbox Vault Name)',
                description='See https://wiki.yandex-team.ru/Sandbox/vault/#web',
                required=True,
                default='light_page_reviser_password',
            )

            need_auth_owner = sdk2.parameters.String(
                'Owner for Sandbox Vault (Login and Password)',
                description='See https://wiki.yandex-team.ru/Sandbox/vault/#web',
                required=True,
                default='VELOCITY'
            )

        with sdk2.parameters.Group('Upload to YT') as yt_params:
            upload_to_yt = sdk2.parameters.Bool('Upload results to YT', default=False)

            with upload_to_yt.value[True]:
                project = sdk2.parameters.String('Project', required=True)

    def on_prepare(self):
        task_dir = os.path.dirname(os.path.realpath(__file__))

        run_process(
            ['sh', os.path.join(task_dir, 'prepare.sh')],
            shell=True,
            log_prefix='prepare'
        )

        if self.Parameters.need_auth:
            import common

            self.Context.headers_file = 'headers.json'

            try:
                owner = self.Parameters.need_auth_owner

                response = requests.post(
                    'https://passport.yandex-team.ru/auth',
                    allow_redirects=False,
                    data={
                        'login': sdk2.Vault.data(owner, self.Parameters.need_auth_login),
                        'password': sdk2.Vault.data(owner, self.Parameters.need_auth_password)
                    }
                )

                self.convert_to_file_headers(response.cookies)

            except common.errors.VaultError:
                raise common.errors.TaskFailure("Vault not found. I don't know what to do next")

    def on_execute(self):
        self.shooting()

        if self.Parameters.upload_to_yt:
            results = self.collect_plain_results()
            self.upload_to_yt(results)

        self.aggregate()

    def shooting(self):
        raise NotImplementedError('need to implement')

    def shoot(self, directory, url):
        slug = re.sub(r'[^A-z0-9]', '', url)
        report_dir = 'reports/{}-{}/{}'.format(slug, hash(url), directory)

        os.makedirs(report_dir, 0o755)

        extra_params = []

        if self.Parameters.platform == 'desktop':
            extra_params.append('--emulated-form-factor=none')

        if self.Parameters.perf_only is True:
            extra_params.append('--perf')

        if self.Parameters.need_auth:
            extra_params.append('--extra-headers=./{}'.format(self.Context.headers_file))

        chrome_flags = " ".join([
            '--headless',
            '--disable-gpu',
            '--disable-default-apps',
            '--disable-account-consistency',
            '--disable-cloud-import',
            '--disable-sync',
            '--ignore-certificate-errors'
        ])

        exception = None
        for _ in xrange(RETRY_ON_FAIL_ATTEMPTS):
            try:
                run_process(
                    [
                        'lighthouse',
                        pipes.quote(url),
                        "--chrome-flags='{}'".format(chrome_flags),
                        '--save-assets',
                        '--save-artifacts',
                        '--output=json',
                        '--output=html',
                        '--output-path={}/result.json'.format(report_dir),
                        ' '.join(extra_params)
                    ],
                    shell=True,
                    log_prefix='lighthouse'
                )
                exception = None
                break
            except Exception as e:
                exception = e

        if exception is not None:
            raise exception

        report = sdk2.ResourceData(LightPageReviserReport(
            self,
            'Lighthouse report for ({})'.format(url),
            report_dir,
            directory=report_dir,
            target=url
        ))

        report.ready()

        return report

    def aggregate(self):
        report = OrderedDict()
        urls = self.Parameters.urls.split('\n')

        for index, url in enumerate(urls):
            report[url] = OrderedDict()

            resources = sdk2.Resource.find(
                resource_type=LightPageReviserReport,
                state='READY',
                task=self,
                attrs=dict(target=url)
            ).limit(self.Parameters.iterations)

            p_datas = self.grouped_data_from_resources(resources)

            for metric in METRICS:
                p_data_metric = filter(bool, p_datas[metric])
                report[url][metric] = []

                if len(p_data_metric) == 0:
                    report[url][metric].append({
                        'slice': 'hits',
                        'measure': {
                            'href': None,
                            'value': 0
                        }
                    })

                    report[url][metric].append({
                        'slice': 'mean',
                        'measure': {
                            'href': None,
                            'value': '-'
                        }
                    })

                    for i, p in enumerate(PERCENTILES):
                        report[url][metric].append({
                            'slice': 'p' + str(p),
                            'measure': {
                                'href': None,
                                'value': '-'
                            }
                        })

                else:
                    report[url][metric].append({
                        'slice': 'hits',
                        'measure': {
                            'href': None,
                            'value': len(p_data_metric)
                        }
                    })

                    metric_mean_value = np.mean(p_data_metric)
                    iterate = p_data_metric.index(min(p_data_metric, key=lambda x: abs(x - metric_mean_value)))

                    report[url][metric].append({
                        'slice': 'mean',
                        'measure': {
                            'href': '{}/result.report.html'.format(list(resources)[iterate].http_proxy),
                            'value': metric_mean_value
                        }
                    })

                    p_data = np.percentile(p_data_metric, PERCENTILES)

                    for i, p in enumerate(PERCENTILES):
                        iterate = p_data_metric.index(min(p_data_metric, key=lambda x: abs(x - p_data[i])))

                        report[url][metric].append({
                            'slice': 'p' + str(p),
                            'measure': {
                                'href': '{}/result.report.html'.format(list(resources)[iterate].http_proxy),
                                'value': p_data[i]
                            }
                        })

        self.Context.report = report

    def collect_plain_results(self):
        result = []
        urls = self.Parameters.urls.split('\n')

        for url in urls:
            resources = sdk2.Resource.find(
                resource_type=LightPageReviserReport,
                state='READY',
                task=self,
                attrs=dict(target=url)
            ).limit(self.Parameters.iterations)

            metrics_data = self.plain_data_from_resources(resources)

            for metrics_data_row in metrics_data:
                result.append(
                    dict(
                        platform=self.Parameters.platform,
                        project=self.Parameters.project,
                        url=url,
                        **metrics_data_row
                    )
                )

        return result

    def grouped_data_from_resources(self, resources):
        data = {metric: [] for metric in METRICS}

        for resource in resources:
            metrics = self.extract_metrics_from_report(resource)

            for metric_name in METRICS:
                if metric_name in metrics:
                    data[metric_name].append(metrics[metric_name])

        return data

    def plain_data_from_resources(self, resources):
        return [self.extract_metrics_from_report(resource) for resource in resources]

    def extract_metrics_from_report(self, report_resource):
        with open(os.path.join(report_resource.directory, 'result.report.json')) as json_data:
            report = json.load(json_data)

            result = {
                'score': report['categories']['performance']['score'] * 100
            }

            for metric_name in AUDITS:
                if report['audits'].get(metric_name):
                    raw_value = report['audits'][metric_name]['numericValue']
                    if raw_value:
                        result[metric_name] = round(raw_value, 4)

            return result

    def upload_to_yt(self, data):
        from yt.wrapper import JsonFormat, YtClient

        yt_client = YtClient(
            proxy='hahn',
            token=sdk2.Vault.data(YT_TOKEN_OWNER, YT_TOKEN_NAME)
        )

        table_name = datetime.now().strftime('%Y-%m-%d')
        table_path = os.path.join(YT_PATH, table_name)

        if not yt_client.exists(table_path):
            yt_client.create('table', table_path, recursive=True)

        yt_client.write_table(
            yt_client.TablePath(table_path, append=True),
            data,
            format=JsonFormat()
        )

    def convert_to_file_headers(self, cookies):
        if len(cookies):
            with open(self.Context.headers_file, 'w') as output:
                result = ['{}={}'.format(k, v) for k, v in cookies.items()]
                output.write(json.dumps({'Cookie': ";".join(result)}))

    @sdk2.header()
    def header(self):
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path), extensions=['jinja2.ext.do'])
        return env.get_template("templates/header.html").render({'report': self.Context.report})
