import hashlib
import logging
import os
import shutil
import subprocess
import tarfile

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.search_velocity.resources import TrencherTestLxcImage, TrencherTestRequirements, TrencherTestPackage


class TrencherBuild(sdk2.Task):
    """
    Create package with Trencher sources and virtualenv with requirements
    """
    class Requirements(sdk2.Requirements):
        cores = 4
        disk_space = 5000

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        commit = sdk2.parameters.String('Commit', required=True)
        flow_launch_id = sdk2.parameters.String('Flow launch ID', required=True)
        arcadia_path = sdk2.parameters.String('Arcadia path', default_value='velocity/trencher')

        _container = sdk2.parameters.Container(
            label='Trencher container',
            resource_type=TrencherTestLxcImage,
            platform='linux_ubuntu_16.04_xenial',
            required=True,
        )

    def on_execute(self):
        self.work_dir = os.getcwd()
        self.trencher_dir = os.path.join(self.work_dir, 'trencher')
        self.venv_dir = os.path.join(self.work_dir, 'venv')

        self.get_sources()
        self.get_requirements()
        self.create_package()

    def get_sources(self):
        arc = Arc(secret_name='ARC_TOKEN', secret_owner='VELOCITY')

        export_dir = os.path.join(self.work_dir, 'export')
        with arc.init_bare() as mount_point:
            res = arc.export(mount_point, self.Parameters.commit, self.Parameters.arcadia_path, export_dir)
            logging.info('Export result: %s', res)

        shutil.move(os.path.join(export_dir, self.Parameters.arcadia_path), self.trencher_dir)
        shutil.rmtree(export_dir)

    def get_requirements(self):
        requirements_file = os.path.join(self.trencher_dir, 'requirements.txt')
        if not os.path.isfile(requirements_file):
            raise RuntimeError('There is no requirements.txt')

        with open(requirements_file) as fp:
            requirements_hash = hashlib.sha256(fp.read()).hexdigest()

        res = sdk2.Resource.find(
            type=TrencherTestRequirements,
            state=ctr.State.READY,
            attrs={'hash': requirements_hash},
        ).order(-sdk2.Resource.id).first()

        if res:
            logging.info('Reuse venv resource %s', res.id)

            with tarfile.open(str(sdk2.ResourceData(res).path), 'r|gz') as tar:
                tar.extractall(self.venv_dir)

            return

        logging.info('Create new venv')

        python2 = subprocess.check_output(['which', 'python2']).strip()
        res = subprocess.check_output(['virtualenv', '--always-copy', '--download', '-p', python2, self.venv_dir])
        logging.debug('Virtualenv run results: %s', res)

        pip_path = os.path.join(self.venv_dir, 'bin', 'pip')
        res = subprocess.check_output([pip_path, 'install', '-r', requirements_file])
        logging.debug('Pip run results: %s', res)

        archive_name = 'venv.tar.gz'
        with tarfile.open(archive_name, 'w|gz') as tar:
            tar.add(self.venv_dir, '.')

        res = TrencherTestRequirements(self, 'Trencher test requirements', archive_name, hash=requirements_hash)
        sdk2.ResourceData(res).ready()

    def create_package(self):
        archive_name = 'trencher.tar.gz'
        with tarfile.open(archive_name, 'w|gz') as tar:
            tar.add(self.trencher_dir, 'trencher')
            tar.add(self.venv_dir, 'venv')

        res = TrencherTestPackage(
            self, 'Trencher package', archive_name,
            commit=self.Parameters.commit,
            flow_launch_id=self.Parameters.flow_launch_id,
        )
        sdk2.ResourceData(res).ready()
