import logging
import os
import subprocess
import tarfile

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects.search_velocity.resources import TrencherTestLxcImage, TrencherTestPackage, TrencherTestLogs
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class TrencherTest(sdk2.Task):
    """
    Run Trencher test script
    """
    class Requirements(sdk2.Requirements):
        cores = 4
        disk_space = 5000

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        commit = sdk2.parameters.String('Commit', required=True)
        flow_launch_id = sdk2.parameters.String('Flow launch ID', required=True)
        test_script = sdk2.parameters.String('Test script', required=True)

        _container = sdk2.parameters.Container(
            label='Trencher container',
            resource_type=TrencherTestLxcImage,
            platform='linux_ubuntu_16.04_xenial',
            required=True,
        )

    class Context(sdk2.Context):
        test_report = ''

    @sdk2.footer()
    def footer(self):
        if not self.Context.test_report:
            return ''

        return '<h3>Test report</h3><pre>%s</pre>' % self.Context.test_report

    def on_execute(self):
        self.work_dir = os.getcwd()

        self.trencher_dir = os.path.join(self.work_dir, 'trencher')
        self.venv_dir = os.path.join(self.work_dir, 'venv')

        self.prepare_package()
        self.run_tests()

    def prepare_package(self):
        package = sdk2.Resource.find(
            type=TrencherTestPackage,
            state=ctr.State.READY,
            attrs={
                'commit': self.Parameters.commit,
                'flow_launch_id': self.Parameters.flow_launch_id,
            }
        ).order(-sdk2.Resource.id).first()

        if not package:
            raise SandboxTaskFailureError('Trencher package for this CI flow was not found')

        package_path = str(sdk2.ResourceData(package).path)
        with tarfile.open(package_path, 'r|gz') as tar:
            tar.extractall(self.work_dir)

        if not os.path.isdir(self.trencher_dir):
            raise RuntimeError('There is no trencher dir in package')

        if not os.path.isdir(self.venv_dir):
            raise RuntimeError('There is no venv dir in package')

    def run_tests(self):
        test_script = os.path.join(self.trencher_dir, self.Parameters.test_script)

        if not os.path.isfile(test_script):
            raise SandboxTaskFailureError('Test script was not found')

        venv_bin = os.path.join(self.venv_dir, 'bin')
        env_path = venv_bin + ':' + os.environ.get('PATH', '')
        logging.debug('Test launch PATH: %s', env_path)

        log_dir = os.path.join(self.work_dir, 'test_logs')
        os.makedirs(log_dir)

        out_log = os.path.join(log_dir, 'test.out.log')
        err_log = os.path.join(log_dir, 'test.err.log')

        with open(out_log, 'w') as o_fp, open(err_log, 'w') as e_fp:
            p = subprocess.Popen(
                [test_script], stdout=o_fp, stderr=e_fp,
                env={'HOME': self.work_dir, 'PATH': env_path},
            )

            ret_code = p.wait()

        commit = str(self.Parameters.commit)
        flow_launch_id = str(self.Parameters.flow_launch_id)

        res = TrencherTestLogs(self, 'Test logs', log_dir, commit=commit, flow_launch_id=flow_launch_id)
        sdk2.ResourceData(res).ready()

        with open(out_log) as fp:
            self.Context.test_report = fp.read().strip()

        if ret_code:
            raise SandboxTaskFailureError('Test run failed with code %s' % ret_code)
