# coding: utf-8

import os
import logging
import time
import subprocess
import random
import string

from sandbox.sdk2.helpers import ProcessLog

from sandbox import sdk2

from sandbox.common.types import misc as ctm
from sandbox.common.types import client
from sandbox.common import errors

from sandbox.projects.impulse.BuildCodeQLIndex import BuildCodeqlIndex
from sandbox.projects.impulse.SastWorkflow.checkout import CheckoutOperation


def lazyprop(fn):
    attr_name = '_lazy_' + fn.__name__

    @property
    def _lazyprop(self):
        if not hasattr(self, attr_name):
            setattr(self, attr_name, fn(self))
        return getattr(self, attr_name)

    return _lazyprop


class BuildCodeqlIndexForToloka(BuildCodeqlIndex):

    __docker_root_name = 'docker_root'

    class Requirements(sdk2.Task.Requirements):
        privileged = True
        dns = ctm.DnsType.DNS64
        client_tags = client.Tag.IPV6 & client.Tag.LINUX_TRUSTY
        disk_space = 4096
        ram = 2048

    class Parameters(BuildCodeqlIndex.Parameters):
        language = sdk2.parameters.String("Language", default="java", required=True)
        dockerfile_dir = sdk2.parameters.String('Path to directory with Dockerfile', default="toloka-qloud-microservices/vagrant/ansible/docker/postgresql-all/", required=True)
        docker_run_opts = sdk2.parameters.String('docker run options', default="-p 5432:5432 -p 6543:6543", required=False)
        src_build_command = sdk2.parameters.String('Command for building sources', default="gradlew build -x test -x integrationTest -x apiator", required=True)

    @lazyprop
    def __task_p(self):
        self.__task_path = os.path.dirname(os.path.realpath(__file__))
        return self.__task_path

    @lazyprop
    def __docker_root_p(self):
        self.__docker_root = os.path.join(
            self.__task_p, self.__docker_root_name)
        try:
            os.makedirs(self.__docker_root)
        except OSError as e:
            if e.errno != 17:
                raise errors.TaskFailure(e)
        return self.__docker_root

    @lazyprop
    def __docker_cmd(self):
        return 'docker'

    @lazyprop
    def _docker_tag(self):
        letters = string.ascii_lowercase + string.digits
        return ''.join(random.choice(letters) for i in range(20))

    def _prepare_docker(self):
        docker_setup_script = os.path.join(
            self.__task_p, 'docker_setup_v6.sh')
        with ProcessLog(self, logger="docker_prepare") as pl:
            cmd = "bash {docker_setup_script} {docker_root}".format(
                docker_setup_script=docker_setup_script,
                docker_root=self.__docker_root_p
            )
            status = subprocess.Popen(
                cmd,
                shell=True,
                stderr=pl.stdout,
                stdout=pl.stdout
            ).wait()
            if status != 0:
                raise Exception("Docker prepare failed")

    def _build_dockerfile(self, target, tag):
        with ProcessLog(self, logger='docker_build') as pl:
            cmd = '{docker_cmd} build -t {tag} {target}'.format(
                docker_cmd=self.__docker_cmd,
                tag=tag,
                target=target
                )
            status = subprocess.Popen(
                cmd,
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stderr).wait()
            if status != 0:
                raise Exception("Docker build failed")

    def _run_docker(self, tag, options):
        with ProcessLog(self, logger='docker_run') as pl:
            cmd = '{docker_cmd} run -d {options} {tag}'.format(
                docker_cmd=self.__docker_cmd,
                options=options,
                tag=tag
            )

            status = subprocess.Popen(
                cmd,
                shell=True,
                stdout=subprocess.PIPE,
                stderr=pl.stderr).wait()
            if status != 0:
                raise Exception("Docker run failed")

    def _stop_docker(self):
        with ProcessLog(self, logger='docker_stop') as pl:
            cmd = '{docker_cmd} stop $({docker_cmd} ps -a -q)'.format(
                docker_cmd=self.__docker_cmd,
                )
            status = subprocess.Popen(
                cmd,
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stderr
            ).wait()
            if status != 0:
                raise Exception("Docker stop failed")

    def on_execute(self):
        language = self.Parameters.language

        # checkout repository
        repository = [{"url": self.Parameters.repo_url}]
        repo = CheckoutOperation(repository)
        with sdk2.ssh.Key(self, key_owner='IMPULSE_SSH_KEY'):
            repo.checkout()
        logging.debug("[+] checkout success: {}.".format(repo.folders))

        # preparing docker
        self._prepare_docker()
        logging.debug("[+] docker environment was prepared")

        # building docker image
        src_path = list(repo.folders.keys())[0]
        docker_build_target = os.path.abspath(os.path.join(src_path, self.Parameters.dockerfile_dir))
        self._build_dockerfile(docker_build_target, self._docker_tag)
        logging.debug("[+] docker image was built")

        # run container
        options = self.Parameters.docker_run_opts
        self._run_docker(self._docker_tag, options)
        time.sleep(30)

        # analyze
        build_command = "{}/{}".format(src_path, self.Parameters.src_build_command)
        self.process_language(language, repo, build_command)
        logging.debug("[+] language was processed: {}".format(language))

        # stop docker
        self._stop_docker()
        logging.debug("[+] docker was stopped")
