# coding: utf-8

import os
import logging

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common import errors
from sandbox.common.types import client as ctc
from sandbox.common.types import misc as ctm
from sandbox.sandboxsdk import environments
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.constants import constants as sdk_constants

from sandbox.projects.security.common import helpers
from sandbox.projects.security.resources import SecurityToolBinary, TvmToolBinary


RESOURSE_TYPES = {
    'SECURITY_TOOL_BINARY': SecurityToolBinary,
    'TVM_TOOL_BINARY': TvmToolBinary,
}


PLATFORM_TO_ARCH = {
    'darwin': 'osx',
    'linux': 'linux',
    'osx': 'osx',
    'windows': 'win_nt',
}


class BuildSecurityToolArc(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('wheel', version='0.30.0'),
            environments.NodeJS('8.9.4'),
        ]
        client_tags = ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        project_name = sdk2.parameters.String('Project name', required=True)
        author = sdk2.parameters.String('Project author (e.g. "Man <man@ya.ru>")', required=True)
        arcadia_url = sdk2.parameters.String('Path to Arcadia svn', default='arcadia:/arc/trunk/arcadia', required=True)
        disable_cgo = sdk2.parameters.Bool('Disable CGO', default=False)
        strip_binaries = sdk2.parameters.Bool('Strip binaries', default=False)
        build_darwin = sdk2.parameters.Bool('Build tool for OSX', default=True)
        build_windows = sdk2.parameters.Bool('Build tool for Windows', default=True)
        version_cmd = sdk2.parameters.String('Version command', default='{tool} --version', required=True)
        do_deploy = sdk2.parameters.Bool('Deploy (upload to Sectools service)', default=False)
        sectools_token = sdk2.parameters.Vault('PyPi password', default_value='SECTOOLS_TOKEN')
        sectools_channel = sdk2.parameters.String('Sectools channel', default='stable')
        deploy_pypi = sdk2.parameters.Bool('Build&Upload pypi package (must have PYPI_PASSWORD in vault)', default=False)
        pypi_login = sdk2.parameters.String('PyPi login')
        pypi_password = sdk2.parameters.Vault('PyPi password', default_value='PYPI_PASSWORD')
        deploy_npm = sdk2.parameters.Bool('Build&Upload npm package', default=False)
        npm_login = sdk2.parameters.String('NPM login')
        npm_password = sdk2.parameters.Vault('NPM password', default='NPM_PASSWORD')
        do_npm_pack_binaries = sdk2.parameters.Bool('Pack binaries into the npm package', default=True)
        with sdk2.parameters.RadioGroup('Type of resourse on finish') as res_type:
            res_type.values['SECURITY_TOOL_BINARY'] = res_type.Value(value='SECURITY_TOOL_BINARY', default=True)
            res_type.values['TVM_TOOL_BINARY'] = res_type.Value(value='TVM_TOOL_BINARY')
        testing_mode = sdk2.parameters.Bool('Testing mode', default=False)

    def on_execute(self):
        platforms = ['linux']

        if self.Parameters.build_darwin:
            platforms.append('darwin')

        if self.Parameters.build_windows:
            platforms.append('windows')

        if not platforms:
            raise errors.TaskFailure('Must me user at leat one platform (linux, darwin or windows)')

        def_flags = {'FORCE_VCS_INFO_UPDATE': 'yes'}
        if self.Parameters.disable_cgo:
            def_flags['CGO_ENABLED'] = '0'

        binaries = {}
        target = sdk2.svn.Arcadia.parse_url(self.Parameters.arcadia_url).subpath
        svn_url = sdk2.svn.Arcadia.svn_url(self.Parameters.arcadia_url)
        arcadia_path = arcadiasdk.do_clone(svn_url, self, use_checkout=False)
        for platform in platforms:
            build_directory = str(self.path(os.path.join('build', platform)))
            logging.info('build for platform: %s' % platform)
            arcadiasdk.do_build(
                build_system=sdk_constants.YMAKE_BUILD_SYSTEM,
                build_type=sdk_constants.RELEASE_BUILD_TYPE,
                source_root=arcadia_path,
                targets=[target],
                results_dir=build_directory,
                target_platform=platform,
                checkout=True,
                clear_build=True,
                strip_binaries=self.Parameters.strip_binaries,
                def_flags=def_flags
            )
            binaries[platform] = self.find_binary(os.path.join(build_directory, 'bin'))
            logging.info('builded at: %s' % binaries[platform])

        version = self.get_version(binaries['linux'])
        project_name = self.Parameters.project_name
        for platform in platforms:
            logging.info('create resource for platform: %s' % platform)
            sdk2.ResourceData(
                RESOURSE_TYPES[self.Parameters.res_type](
                    self,
                    '%s for %s (%s)' % (project_name, platform.title(), version),
                    binaries[platform],
                    arch=PLATFORM_TO_ARCH.get(platform, 'linux'),
                    ttl='inf',
                    version=version,
                    target=target
                )
            )

        if self.Parameters.do_deploy:
            logging.info('deploy tool to sectools')
            self.deploy_sectools(
                testing_mode=self.Parameters.testing_mode,
                project_name=project_name,
                version=version,
                binaries=binaries,
            )

        if self.Parameters.deploy_pypi:
            logging.info('deploy to pypi')
            self.deploy_pypi(project_name, version, platforms)

        if self.Parameters.deploy_npm:
            logging.info('deploy to npm')
            self.deploy_npm(project_name, version, platforms)

    def deploy_sectools(self, testing_mode, project_name, version, binaries):
        tool_name = project_name.lower()
        releaser = self.download_relaser()
        channel = self.Parameters.sectools_channel
        if not channel:
            channel = 'stable'

        upstream = 'https://tools.sec.yandex-team.ru'
        if testing_mode:
            upstream = 'https://tools-test.sec.yandex-team.ru'

        env = os.environ.copy()
        env['SECTOOLS_TOKEN'] = sdk2.Vault.data(self.Parameters.sectools_token)

        args = [
            releaser,
            'tools',
            'upload',
            '--upstream', upstream,
            '--name', tool_name,
            '--version', version,
            '--release-to', channel,
        ]
        for platform, binary in binaries.items():
            args += ['%s:%s' % (platform, binary)]

        self.run_cmd(
            args,
            log_prefix='sectools_deploy',
            environment=env
        )

    def deploy_pypi(self, project_name, version, platforms):
        env = os.environ.copy()
        env['PYPI_LOGIN'] = self.Parameters.pypi_login
        env['PYPI_PASSWORD'] = sdk2.Vault.data(self.Parameters.pypi_password)

        args = [
            os.path.join(os.path.abspath(os.path.dirname(__file__)), 'pip', 'build'),
            '--version', version,
            '--author', self.Parameters.author,
            '--platforms', ','.join(platforms),
            '--workdir', str(self.path('pypi')),
            project_name.lower()
        ]
        self.run_cmd(
            args,
            log_prefix='pypi_deploy',
            environment=env
        )

    def deploy_npm(self, project_name, version, platforms):
        env = os.environ.copy()
        env['NPM_LOGIN'] = self.Parameters.npm_login
        env['NPM_PASSWORD'] = sdk2.Vault.data(self.Parameters.npm_password)

        args = [
            os.path.join(os.path.abspath(os.path.dirname(__file__)), 'npm', 'build'),
            '--version', version,
            '--author', self.Parameters.author,
            '--platforms', ','.join(platforms),
            '--workdir', str(self.path('npm')),
            project_name.lower()
        ]
        if self.Parameters.do_npm_pack_binaries:
            args.append('--pack-binaries')

        self.run_cmd(
            args,
            log_prefix='npm_deploy',
            environment=env
        )

    def run_cmd(self, cmd, log_prefix, environment):
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(log_prefix)) as pl:
            sp.check_call(cmd, env=environment, stdout=pl.stdout, stderr=pl.stderr)

    def get_version(self, tool):
        cmd_path = str(self.path('version.sh'))
        with open(cmd_path, 'wt') as cmd_file:
            cmd_file.write(self.Parameters.version_cmd.format(tool=tool))

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('get_version')) as pl:
            version = sp.check_output(['/bin/bash', cmd_path], stderr=pl.stderr)
            return version.strip()

    def find_binary(self, bin_path):
        def is_tool(file_name):
            return not os.path.isdir(os.path.join(bin_path, file_name))

        tool_directory_file_names = [x for x in os.listdir(bin_path) if is_tool(x)]
        if len(tool_directory_file_names) != 1:
            raise errors.TaskFailure(
                'Expected to have exactly one binary for tool, path {path}, but got: {names}'.format(path=bin_path, names=tool_directory_file_names)
            )
        return os.path.join(bin_path, tool_directory_file_names[0])

    def download_relaser(self):
        resource = helpers.get_last_released_resource(
            resource_type=SecurityToolBinary,
            attrs={
                'target': 'security/sectools/cmd/releaser',
            }
        )

        binary = helpers.download_resource(resource)
        helpers.chmod_x(binary)
        return binary
