# coding: utf-8
import os
import logging
import tempfile

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.common.types import client as ctc
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.sandboxsdk import environments
from sandbox.projects import resource_types
from sandbox.projects.security.common.task import SecurityBaseTask


class CheckPortoLayerSecurity(SecurityBaseTask):
    class Requirements(sdk2.Task.Requirements):
        container_resource = 1588215291
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 16384
        ram = 4096
        cores = 4
        environments = [
            environments.PipEnvironment('startrek_client', use_wheel=True),
        ]

    class SecurityOptions(SecurityBaseTask.SecurityOptions):
        subtask_resource_attrs = {
            "target": "security/yadi/yadi-os/cmd/yadi-os"
        }

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup("Minimum severity") as severity:
            severity.values["-l"] = severity.Value(value="Low")
            severity.values["-ll"] = severity.Value(value="Medium", default=True)
            severity.values["-lll"] = severity.Value(value="High")
            severity.values["-llll"] = severity.Value(value="Critical")
        layer_types = sdk2.parameters.List("Types of layer resources", required=True)
        create_st_ticket = sdk2.parameters.Bool("Create ST ticket", default=True)
        st_queue = sdk2.parameters.String("ST queue key", default="HOSTMAN", required=True)
        st_followers = sdk2.parameters.String("ST followers", default="buglloc", required=True)
        st_components = sdk2.parameters.List("ST components")

    def make_yadi_os_cmd(self, binary_path, root):
        cmd = [
            binary_path,
            "test", "packages",
            self.Parameters.severity,
            "--root", root,
            "--exit-code=3",
        ]

        return cmd

    def on_execute(self):
        for resource_type in self.Parameters.layer_types:
            layer_id, workdir = self.prepare_resource(resource_type)
            out_path = str(self.path("yadi-os-%s.txt" % layer_id))
            returncode = 0
            with open(out_path, "wt") as fd:
                with sdk2.helpers.ProcessLog(self, logger="yadi-os-%s" % layer_id) as pl:
                    cmd = self.make_yadi_os_cmd(self.subtask_binary_path, workdir)
                    env = self.get_env()
                    p = subprocess.Popen(cmd, shell=False, stdout=fd, stderr=pl.stderr, env=env)
                    p.wait()
                    returncode = p.returncode

            if returncode == 0:
                self.set_info("Layer %s analyzed: no issues found" % layer_id)
                logging.info("Layer %s analyzed: no issues found" % layer_id)
                continue
            elif returncode != 3:
                raise Exception("unexpected yadi-os exit code: %d", returncode)

            logging.info("Layer %s analyzed: issues saved into: %s" % (layer_id, out_path))
            additional_info = ""
            if self.Parameters.create_st_ticket:
                ticket_id = self.create_ticket(layer_id, out_path)
                if ticket_id:
                    additional_info = ' Ticket: <a href="https://st.yandex-team.ru/{ticket_id}" target="_blank">{ticket_id}</a>'.format(
                        ticket_id=ticket_id,
                    )

            out_res = self.create_resource(layer_id, out_path)
            self.set_info(
                'Layer {layer_id} analyzed. Issues: <a href="{results_uri}" target="_blank">{results_uri}</a>.{additional_info}'.format(
                    layer_id=layer_id,
                    results_uri=out_res.http_proxy,
                    additional_info=additional_info,
                ),
                do_escape=False
            )

    def prepare_resource(self, resource_type):
        logging.info("Prepate porto layer resource, type: %s" % resource_type)
        resource = sdk2.Resource[resource_type].find(
            attrs=dict(released=ctt.ReleaseStatus.STABLE),
            state=ctr.State.READY,
        ).order(-sdk2.Resource.id).first()

        layer_id = "%s@%d" % (resource_type, resource.id)
        layer_path = sdk2.ResourceData(resource).path.as_posix()
        logging.info("Downloaded layer {layer_id} into: {layer_path}".format(layer_id=layer_id, layer_path=layer_path))

        workdir = tempfile.NamedTemporaryFile(suffix="_%s" % layer_id).name
        os.mkdir(workdir)
        with sdk2.helpers.ProcessLog(self, logger="unpack-%s" % layer_id) as pl:
            cmd = [
                "tar",
                "--exclude=mnt",
                "--exclude=dev",
                "--exclude=proc",
                "--exclude=tmp",
                "--exclude=run",
                "--exclude=sys",
                "--verbose",
                "--extract"
            ]
            cmd.extend(tar_options(layer_path))
            cmd.extend([
                "--file", layer_path,
                "--directory", workdir
            ])

            subprocess.check_call(cmd, shell=False, stderr=pl.stderr)
        logging.info("Layer {layer_id} extracted into: {workdir}".format(layer_id=layer_id, workdir=workdir))
        return (layer_id, workdir)

    def create_resource(self, layer_id, result_path):
        resource = resource_types.OTHER_RESOURCE(
            self,
            "YadiOS results for layer {layer_id}".format(layer_id=layer_id),
            result_path,
        )
        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()
        return resource

    def create_ticket(self, layer_id, result_path):
        from startrek_client import Startrek

        alert_id = '%s@porto-layer' % layer_id
        access_token = sdk2.Vault.data("YADI_ST_TOKEN")
        client = Startrek(useragent="YadiOS monitor", base_url="https://st-api.yandex-team.ru", token=access_token)
        issues = client.issues.find(
            filter={
                "queue": self.Parameters.st_queue,
                "alertID": alert_id,
                "security": "Yes",
                "resolution": "empty()",
            },
            per_page=1,
        )
        if issues:
            logging.info("Skip ST ticket creating for layer %s: ticket already exists" % layer_id)
            # we already have ticket for this layer
            return ""

        desc = ""
        with open(result_path, 'r') as f:
            desc = f.read()

        if not desc:
            raise Exception("emmpty description for layer_id: %s" % layer_id)

        desc = """
Sandbox resource: https://sandbox.yandex-team.ru/resource/{resource_id}/view
-------------------
{desc}
-------------------
Created by task: https://sandbox.yandex-team.ru/task/{task_id}/view
        """.format(
            resource_id=layer_id.split('@').pop(),
            desc=desc,
            task_id=self.id,
        )
        ticket = client.issues.create(
            queue=self.Parameters.st_queue,
            components=self.Parameters.st_components,
            alertID=alert_id,
            security="Yes",
            summary="В порто-слое {layer_id} обнаружены уязвимости.".format(layer_id=layer_id),
            description=desc,
            followers=[x.strip() for x in self.Parameters.st_followers.split(",")],
        )
        logging.info("ticket for layer_id {layer_id} created: {ticket_key}".format(layer_id=layer_id, ticket_key=ticket.key))
        return ticket.key


def tar_options(filename):
    options = {
        ".xz": ["--xz"],
        ".txz": ["--xz"],
        ".gz": ["--gzip"],
        ".tgz": ["--gzip"],
        ".zst": ["--use-compress-program", "zstd"],
        ".tzst": ["--use-compress-program", "zstd"],
    }

    _, ext = os.path.splitext(filename)
    if ext in options:
        return options[ext]
    else:
        logging.warn("unknown compression: %s", filename)
        return []
