# coding: utf-8
import os
import logging
import tempfile

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.common.types import client as ctc
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import ssh
from sandbox.projects import resource_types
from sandbox.projects.security.common.task import SecurityBaseTask


class CheckRtcSaltSecurity(SecurityBaseTask):
    class Requirements(sdk2.Task.Requirements):
        container_resource = 1598242028
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 16384
        ram = 4096
        cores = 4
        environments = [
            environments.PipEnvironment('startrek_client', use_wheel=True),
        ]

    class SecurityOptions(SecurityBaseTask.SecurityOptions):
        subtask_resource_attrs = {
            "target": "security/yadi/yadi-os/cmd/yadi-os"
        }

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup("Minimum severity") as severity:
            severity.values["-l"] = severity.Value(value="Low")
            severity.values["-ll"] = severity.Value(value="Medium", default=True)
            severity.values["-lll"] = severity.Value(value="High")
            severity.values["-llll"] = severity.Value(value="Critical")
        repo = sdk2.parameters.String("saltstack repo", default="ssh://git@bb.yandex-team.ru/rtcsalt/core.git", required=True)
        clusters = sdk2.parameters.List("clusters", default="sas", required=True)
        create_st_ticket = sdk2.parameters.Bool("Create ST ticket", default=True)
        st_queue = sdk2.parameters.String("ST queue key", default="HOSTMAN", required=True)
        st_followers = sdk2.parameters.String("ST followers", default="buglloc", required=True)
        st_components = sdk2.parameters.List("ST components")

    def make_yadi_os_cmd(self, binary_path, cluster, root):
        cmd = [
            binary_path,
            "hostctl",
            self.Parameters.severity,
            "--cluster", cluster,
            "--unit", os.path.join(root, "units.d/upstream-packages.yaml"),
            "--exit-code=3",
        ]

        return cmd

    def on_execute(self):
        workdir = tempfile.NamedTemporaryFile().name
        os.mkdir(workdir)
        with ssh.Key(self, self.owner, 'YADI_SSH_KEY'):
            with sdk2.helpers.ProcessLog(self, logger="git-clone") as pl:
                subprocess.check_call(["git", "clone", self.Parameters.repo, workdir], shell=False, stdout=pl.stdout, stderr=pl.stderr)

        os.chdir(workdir)
        vulns = []
        for cluster in self.Parameters.clusters:
            subprocess.check_call(["git", "checkout", cluster], shell=False)
            rev = subprocess.check_output(["git", "rev-parse", "HEAD"], shell=False).strip()

            out_path = str(self.path("yadi-os-%s.txt" % cluster))
            returncode = 0
            with open(out_path, "wt") as fd:
                with sdk2.helpers.ProcessLog(self, logger="yadi-os-%s" % cluster) as pl:
                    cmd = self.make_yadi_os_cmd(self.subtask_binary_path, cluster, workdir)
                    env = self.get_env()
                    p = subprocess.Popen(cmd, shell=False, stdout=fd, stderr=pl.stderr, env=env)
                    p.wait()
                    returncode = p.returncode

            if returncode == 0:
                self.set_info("Branch %s analyzed: no issues found" % cluster)
                logging.info("Branch %s analyzed: no issues found" % cluster)
                continue
            elif returncode != 3:
                raise Exception("unexpected yadi-os exit code: %d", returncode)

            logging.info("Cluster %s analyzed: issues saved into: %s" % (cluster, out_path))
            vulns.append({"cluster": cluster, "rev": rev, "info_path": out_path})
            out_res = self.create_resource(cluster, rev, out_path)
            self.set_info(
                'Cluster {cluster} analyzed. Issues: <a href="{results_uri}" target="_blank">{results_uri}</a>.'.format(
                    cluster=cluster,
                    results_uri=out_res.http_proxy,
                ),
                do_escape=False
            )

        if self.Parameters.create_st_ticket and vulns:
            self.create_ticket(vulns)

    def create_resource(self, branch, rev, result_path):
        resource = resource_types.OTHER_RESOURCE(
            self,
            "YadiOS results for RTCSALT {branch}@{rev}".format(branch=branch, rev=rev),
            result_path,
        )
        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()
        return resource

    def create_ticket(self, vulns):
        from startrek_client import Startrek

        alert_id = 'hostctl'
        access_token = sdk2.Vault.data("YADI_ST_TOKEN")
        client = Startrek(useragent="YadiOS monitor", base_url="https://st-api.yandex-team.ru", token=access_token)
        issues = client.issues.find(
            filter={
                "queue": self.Parameters.st_queue,
                "alertID": alert_id,
                "security": "Yes",
                "resolution": "empty()",
            },
            per_page=1,
        )
        if issues:
            logging.info("Skip ST ticket creating: ticket already exists")
            return ""

        desc = []
        for vuln in vulns:
            desc.append("Cluster: {cluster}    Rev: {rev}".format(cluster=vuln["cluster"], rev=vuln["rev"]))
            desc.append("-------------------")
            with open(vuln["info_path"], "r") as f:
                info = f.read()
                if not info:
                    raise Exception("empty description for cluster: {cluster}".format(cluster=vuln["cluster"]))
                desc.append(info)
            desc.append("-------------------")
        desc.append("Created by task: https://sandbox.yandex-team.ru/task/{task_id}/view".format(task_id=self.id))

        ticket = client.issues.create(
            queue=self.Parameters.st_queue,
            components=self.Parameters.st_components,
            alertID=alert_id,
            security="Yes",
            summary="Обнаружены уязвимые пакетики",
            description="\n".join(desc),
            followers=[x.strip() for x in self.Parameters.st_followers.split(",")],
        )
        logging.info("ticket created: {ticket_key}".format(ticket_key=ticket.key))
        return ticket.key
