import os
import logging

from sandbox import sdk2
import sandbox.common.types.misc as ctm


class CodeQLAnalysisResource(sdk2.Resource):
    """ Plaintext file resource """


class CodeQLArcadia(sdk2.Task):

    CODEQL_HOME = "/opt/codeql-home"
    CODEQL_DATABASES = "/opt/codeql-home/databases"
    CODEQL_ANALYSIS = "/opt/codeql-home/analysis"
    CODEQL_CLI = "/opt/codeql-home/codeql-cli/codeql"
    CODEQL_STUB_CPP = "/opt/codeql-home/stubs/cpp"
    ARCADIA_HOME = "/opt/arcadia"

    class Requirements(sdk2.Task.Requirements):
        container_resource = 1610503247     # 1488529073
        privileged = True

    class Parameters(sdk2.Task.Parameters):
        yav_secrets = sdk2.parameters.YavSecret("Yav secret with private ssh key (to go to arcadia) and ya_token, ya_user", default="sec-01e785veywb6ztcqgvpf5fdvte@ver-01e7ap91rcv0j941ns3ej0kv2n")
        path_to_project = sdk2.parameters.String("path to arcadia project", default="contrib/libs/zlib")
        language = sdk2.parameters.String("analysis language", default="cpp")
        codeql_suite = sdk2.parameters.String("run queries from cpecified codeql suite", default="cpp-lgtm-full.qls")
        suspend_on_start = sdk2.parameters.Bool('Suspend task on start', default=False)

    def execute_with_logging(self, cmd, shell=False):
        import subprocess
        s = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=shell)
        logging.debug("[+] pid %s: started: %s", os.getpid(), str(cmd))
        stdout, stderr = s.communicate()
        exitcode = s.returncode
        logging.debug("[+] pid %s: stdout: %s", os.getpid(), stdout)
        logging.debug("[+] pid %s: stderr: %s", os.getpid(), stderr)
        logging.debug("[+] pid %s: exitcode: %d", os.getpid(), exitcode)
        logging.debug("[+] pid %s: finished: %s", os.getpid(), str(cmd))
        return exitcode

    def on_execute(self):
        """Main sanbox task excution task."""

        if self.Parameters.suspend_on_start:
            self.suspend()

        # Add go binary to PATH
        os.environ["PATH"] += os.pathsep + "/usr/local/go/bin"

        self.execute_with_logging(["id"])
        self.execute_with_logging(["pwd"])
        self.execute_with_logging(["printenv"])

        logging.debug("[+] Setting YA_TOKEN & YA_USER")
        os.environ["YA_TOKEN"] = self.Parameters.yav_secrets.data()["ya_token"]
        os.environ["YA_USER"] = self.Parameters.yav_secrets.data()["ya_user"]
        logging.debug("[+] YA_USER = {}".format(os.environ["YA_USER"]))

        logging.debug("[+] Downloading Arcadia base")
        self.execute_with_logging("svn cat svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/ya | python - clone {}".format(self.ARCADIA_HOME), shell=True)

        logging.debug("[+] Change Workdir")
        os.chdir(self.ARCADIA_HOME)

        logging.debug("[+] Block ms requests by codeql cli")
        self.execute_with_logging('echo "127.0.0.1 vortex.data.microsoft.com" >> /etc/hosts', shell=True)

        logging.debug("[+] Checkouting target project")
        # with sdk2.ssh.Key(private_part=self.Parameters.yav_secrets.data()["ssh_private_key"]):
        self.execute_with_logging(["{}/ya".format(self.ARCADIA_HOME), "make", "--checkout", "-j0", self.Parameters.path_to_project])

        if self.Parameters.language in ["javascript", "python", "go"]:
            logging.debug("[+] Create database for non-compiled language")
            cmd = '{} database create --language={} --source-root={}/{} {}/database.codeqldb'.format(
                self.CODEQL_CLI, self.Parameters.language, self.ARCADIA_HOME, self.Parameters.path_to_project, self.CODEQL_DATABASES
            )
        else:
            logging.debug("[+] Create database for compiled language")
            cmd = '{} database create --language={} --source-root={}/{} --command="{}/ya make" {}/database.codeqldb'.format(
                self.CODEQL_CLI, self.Parameters.language, self.ARCADIA_HOME, self.Parameters.path_to_project, self.ARCADIA_HOME, self.CODEQL_DATABASES
            )
        self.execute_with_logging(cmd, shell=True)

        logging.debug("[+] Upgrade database")
        cmd = [self.CODEQL_CLI, "database", "upgrade", "{}/database.codeqldb".format(self.CODEQL_DATABASES)]
        self.execute_with_logging(cmd)

        results_file = "{}/result.scarif.json".format(self.CODEQL_ANALYSIS)

        logging.debug("[+] Analyse database")
        cmd = [
            self.CODEQL_CLI, "database", "analyze", "{}/database.codeqldb".format(self.CODEQL_DATABASES),
            self.Parameters.codeql_suite, "--format=sarifv2.1.0", "--output={}".format(results_file)
        ]
        self.execute_with_logging(cmd)

        f = open(results_file, "r")
        results = f.read()
        f.close()

        resource = sdk2.ResourceData(CodeQLAnalysisResource(
            self, "Analysis results file", "results.scarif.json"
        ))
        resource.path.write_bytes(results)
        resource.ready()


class CodeQLArcadiaScheduler(sdk2.Task):
    def on_execute(self):
        import os
        import random
        import subprocess

        random.seed(os.urandom(10))

        cmd = 'svn list -R svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/contrib/libs | grep -P "ya.make"'
        logging.info("[+] Run: {}".format(cmd))
        r = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE)
        output = r.communicate()[0]

        lines = output.split()
        lines = list(filter(lambda x: x != "ya.make", lines))

        project_path = "contrib/libs/{}".format(random.choice(lines).decode())
        project_path = project_path[:-8]
        logging.info("[+] Selected project_path: {}".format(project_path))

        child = CodeQLArcadia(
            self,
            description="Child of Scheduler {}".format(self.id),
            owner="procenkoeg",
            path_to_project=project_path
        )
        child.enqueue()


class CodeqlArcadiaPythonContribScheduler(sdk2.Task):
    def on_execute(self):
        import os
        import random
        import subprocess

        random.seed(os.urandom(10))

        cmd = 'svn list -R svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/contrib/python | grep -P "ya.make"'
        logging.info("[+] Run: {}".format(cmd))
        r = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE)
        output = r.communicate()[0]

        lines = output.split()
        lines = list(filter(lambda x: x != "ya.make", lines))

        project_path = "contrib/python/{}".format(random.choice(lines).decode())
        project_path = project_path[:-8]
        logging.info("[+] Selected project_path: {}".format(project_path))

        child = CodeQLArcadia(
            self,
            description="Child of Python Contrib Scheduler {}".format(self.id),
            owner="procenkoeg",
            path_to_project=project_path,
            language="python",
            codeql_suite="python-lgtm-full.qls"
        )
        child.enqueue()



class CodeqlArcadiaGoContribScheduler(sdk2.Task):

    def on_execute(self):
        import os
        import random
        import subprocess

        random.seed(os.urandom(10))

        cmd = 'svn list -R svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/vendor | grep -P "ya.make"'
        logging.info("[+] Run: {}".format(cmd))
        r = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE)
        output = r.communicate()[0]

        lines = output.split()
        lines = list(filter(lambda x: x != "ya.make", lines))

        project_path = "vendor/{}".format(random.choice(lines).decode())
        project_path = project_path[:-8]
        logging.info("[+] Selected project_path: {}".format(project_path))

        child = CodeQLArcadia(
            self,
            description="Child of Go Contrib Scheduler {}".format(self.id),
            owner="procenkoeg",
            path_to_project=project_path,
            language="go",
            codeql_suite="go-lgtm-full.qls"
        )
        child.enqueue()


class CodeQLTestNoInternet(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        container_resource = 1610503247     # 1488529073
        privileged = True
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        suspend_on_start = sdk2.parameters.Bool('Suspend task on start', default=False)
        suspend_on_finish = sdk2.parameters.Bool('Suspend task on finish', default=False)

    def execute_with_logging(self, cmd, shell=False):
        import subprocess
        s = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=shell)
        logging.debug("[+] pid %s: started: %s", os.getpid(), str(cmd))
        stdout, stderr = s.communicate()
        exitcode = s.returncode
        logging.debug("[+] pid %s: stdout: %s", os.getpid(), stdout)
        logging.debug("[+] pid %s: stderr: %s", os.getpid(), stderr)
        logging.debug("[+] pid %s: exitcode: %d", os.getpid(), exitcode)
        logging.debug("[+] pid %s: finished: %s", os.getpid(), str(cmd))
        return exitcode

    def on_execute(self):
        """Main sanbox task excution task."""

        if self.Parameters.suspend_on_start:
            self.suspend()

        self.execute_with_logging(["id"])
        self.execute_with_logging(["pwd"])
        self.execute_with_logging(["ip", "route"])
        self.execute_with_logging(["ip", "-6", "route"])
        self.execute_with_logging(["ip", "addr"])

        self.execute_with_logging("wget --timeout=10 google.com", shell=True)
        self.execute_with_logging("wget --timeout=10 github.com", shell=True)

        self.execute_with_logging("ip -6 route flush ::/0", shell=True)
        self.execute_with_logging("ip route flush 0/0", shell=True)

        self.execute_with_logging(["ip", "route"])
        self.execute_with_logging(["ip", "-6", "route"])

        self.execute_with_logging("wget --timeout=10 google.com", shell=True)
        self.execute_with_logging("wget --timeout=10 github.com", shell=True)

        if self.Parameters.suspend_on_finish:
            self.suspend()
