# coding: utf-8
from os.path import join, dirname

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.common.types import client as ctc
from sandbox.projects.security.common.task import SecurityBaseTask
from sandbox.projects.security.common import helpers
import shutil

# Configs
config_filename = 'config.json'
working_dir = dirname(__file__)
feed_filename = 'cpp.json.gz'
repo_target_path = 'junk/ya-andrei/cpp-dep-checker'


class FetchNvdFeed(SecurityBaseTask):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 4096
        ram = 2048
        dns = ctm.DnsType.DNS64

    class SecurityOptions(SecurityBaseTask.SecurityOptions):
        subtask_resource_attrs = {
            'target': repo_target_path
        }
        cmd = [
            '-export', feed_filename,
            '-config', config_filename,
            '-gzip',
        ]

    @staticmethod
    def load_local_config(rw_path):
        config_filepath = join(working_dir, config_filename)
        shutil.copy(config_filepath, join(rw_path, config_filename))

    @staticmethod
    def load_config_from_res(rw_path):
        res = helpers.get_last_released_resource(
            ConfigResource,
            None
        )
        if res is None:
            raise Exception('No resource!')
        config = str(res.path.read_bytes())
        with open(join(rw_path, config_filename), 'w') as f:
            f.write(config)

    def on_execute(self):
        rw_path = '.'  # ends with build/linux/bin
        try:
            FetchNvdFeed.load_config_from_res(rw_path)
            print('Config loaded from resources')
        except Exception as e:
            print(e)
            FetchNvdFeed.load_local_config(rw_path)
            print('Loaded local config')
        super(FetchNvdFeed, self).on_execute()
        self.create_resource(feed_filename)
        self.push_config(join(rw_path, config_filename))

    def create_resource(self, filepath):
        res = sdk2.ResourceData(NvdYadiFeed(
            self,
            'NVD feed for Yadi',
            filepath,
            ttl=180
        ))
        res.ready()

    def push_config(self, filepath):
        res = sdk2.ResourceData(ConfigResource(
            self,
            'NVD feed fetcher config',
            filepath
        ))
        res.ready()


class ConfigResource(sdk2.Resource):
    pass


class NvdYadiFeed(sdk2.Resource):
    pass
