import os
import json
import logging
import tempfile
import subprocess
from sandbox import sdk2
from sandbox.projects.security.NirvanaSecCheck.common.utils import cached_property, get_cur_timestamp
from sandbox.projects.security.NirvanaSecCheck.common import http
from sandbox.projects.stop_leak.common import sectools
from sandbox.projects.stop_leak.common.yahec import HecSender


class NirvanaSecCheck(sdk2.Task):
    """ Ant binary secret search task """

    rootfs_dir = ''
    page = 1

    class Parameters(sdk2.Task.Parameters):
        robot_check = sdk2.parameters.Bool('Robot secret owner checker', default=True)
        send_to_splunk = sdk2.parameters.Bool('Send ant-secret results to splunk', default=False)
        default_user = sdk2.parameters.String("Default user for alert with errors in the task", default="shikari", required=True)

    @cached_property
    def temp_dir(self):
        return tempfile.mkdtemp()

    @cached_property
    def ant_path(self):
        """ Download ant-secret resource and return path to it. """
        logging.info("[+] Download Ant secret binary")
        return sectools.download_secret_search()

    @cached_property
    def ant_cmd(self):
        """ Prepare ant-secret command args. """

        _ant_cmd = [self.ant_path, '--status-code=0', '--format=json', '--validate', '--valid-only']
        return _ant_cmd

    def _run_ant_secret(self):
        """Run ant-secret on self.rootfs_dir folder and returns findings."""

        with sdk2.helpers.ProcessLog(self, logger="ant_cmd") as pl:
            cmd = self.ant_cmd + [self.rootfs_dir]
            p = subprocess.Popen(cmd, shell=False, stdout=subprocess.PIPE, stderr=pl.stderr)
            p.wait()
            res = p.communicate()[0]

        if p.returncode:
            raise Exception("Ant Secret Error")

        json_res = json.loads(res)

        return json_res

    def _convert_ant_secret_results_to_splunk_format(self, results):
        """Process and convert ant secret results to splunk compatible format."""

        splunk_results = list()
        for res in results:
            for secret in res.get("secrets", list()):
                splunk_result = {
                    "event": "public secret detected",
                    "time": get_cur_timestamp(),
                    "path": res["path"],
                    "validator": "ant_secret",
                    "line_no": secret["line_no"],
                    "validated": secret["validated"],
                    "secret_type": secret["type"],
                    "secret": secret["secret"],
                    "secret_owner": secret["secret_owner"],
                    "secret_information": secret["additional"]
                }

                if "robot_user" in secret["additional"]:
                    splunk_result["is_robot"] = "True"

                splunk_results.append(splunk_result)

        return splunk_results

    def _send_results_to_splunk(self, splunk_results):
        """Process ant secret results and send them to splunk."""
        logging.info("[+] Send results to Splunk")
        with HecSender(sdk2.Vault.data('STOP_LEAK_NIRVANA_SPLUNK')) as hec_sender:
            for splunk_result in splunk_results:
                hec_sender.send(**splunk_result)

    def get_prefix_weight(self, owner):
        """Staff list sort"""
        prefix_weights = {
            "staff": 0,
            # others 1
            "abc": 2,
        }
        prefix = owner.split(":", 1)[0]
        return prefix_weights.get(prefix, 1)

    def robot_checker(self, ant_secret_results):
        """ Method to check robot logins with ya-resolver """
        session = http.requests_retry_session()
        for chapter in ant_secret_results:

            for users in chapter["secrets"]:

                if not users["owners"]:
                    logging.info("Ant Secret owners is null")
                    users.update({'secret_owner': [self.Parameters.default_user]})
                    continue

                owners = sorted(users["owners"], key=self.get_prefix_weight)

                logging.info("[+] Check login with a resolver: " + str(owners[0].split(':', 1)[1]))
                rsp = session.get("http://ya-resolve.sec.yandex.net/api/v1/resolve?success_only=1&value=" + str(owners[0]) + "&type=person")

                if rsp.status_code != 200:
                    raise Exception("Resolver Api Error " + str(rsp.status_code))

                rsp_json = rsp.json()

                if rsp_json["result"] is None:
                    logging.info("Resolver Api Error: Result is null")
                    users.update({'secret_owner': [self.Parameters.default_user]})
                    continue

                if rsp_json["result"][0]["flow"][0]['resolution'] == "Owner of robot":
                    logging.info("Robot: " + str(owners[0].split(':', 1)[1]))

                    users["additional"].update({'robot_user': [owners[0].split(':', 1)[1]]})

                    users_list = [_['value'] for _ in rsp_json["result"]]
                    users.update({'secret_owner': users_list})

                # Kludge for procenkoeg
                elif rsp_json["result"][0]["flow"][0]['resolution'] == "Resolved":
                    logging.info('Skip a human ' + str(owners[0].split(':', 1)[1]))
                    users.update({'secret_owner': [owners[0].split(':', 1)[1]]})
                else:
                    raise Exception("Resolver Api Error: Unexpected resolution")

        return ant_secret_results

    def file_result(self, rsp, page):
        """ Method to write request result to file from Nirvana API """
        path = os.path.join(self.rootfs_dir, str(page))

        logging.info("[+] Write request result to a file: " + str(path))

        with open(path, 'wt') as file:
            json.dump(rsp.json(), file, indent=4, sort_keys=True)

    def make_data(self, page):
        """ Method to make request to Nirvana API /findSecretKeys"""
        logging.info("[+] Make request to Nirvana API")
        logging.info("Pagination page is " + str(page))

        session = http.requests_retry_session()
        url = "https://nirvana.yandex-team.ru/api/front/findSecretKeys"
        session.headers.update({'Authorization': 'OAuth ' + sdk2.Vault.data('STOP_LEAK_NIRVANA_TOKEN')})

        rsp = session.post(url, json={"id": 3, "jsonrpc": "2.0", "method": "findSecretKeys",
                                      "params": {"availableOnly": "false", "allowedForAnyUserOnly": "false",
                                                 "paginationData": {"pageNumber": str(page), "pageSize": 1000}}})

        if rsp.status_code != 200:
            raise Exception("Nirvana Api Error " + str(rsp.status_code))
        return rsp

    def nirvana_process(self):
        """ Loop to make requests from Nirvana API"""

        rsp = self.make_data(self.page)
        plen = len(rsp.json().get('result'))

        self.rootfs_dir = self.temp_dir + "/http"
        os.makedirs(self.rootfs_dir)

        self.file_result(rsp, self.page)

        while plen >= 1000:
            self.page += 1

            rsp = self.make_data(self.page)

            self.file_result(rsp, self.page)
            plen = len(rsp.json().get('result'))
        return rsp

    def on_execute(self):
        """Main sanbox task excution task."""
        logging.info("[+] Start the task")

        self.nirvana_process()
        ant_secret_results = self._run_ant_secret()

        if self.Parameters.robot_check:
            logging.info("[+] Start a robot check")
            ant_secret_results = self.robot_checker(ant_secret_results)

        logging.info("---------------------- Result --------------------------")
        logging.info(ant_secret_results)

        final_result = self._convert_ant_secret_results_to_splunk_format(ant_secret_results)

        logging.info("--------------------------------------------------------")
        logging.info("[+] Splunk results: {}".format(json.dumps(final_result)))

        if self.Parameters.send_to_splunk:
            self._send_results_to_splunk(final_result)
        logging.info("[+] Stop the task")
