# Notes on market report fuzzing:
# https://wiki.yandex-team.ru/users/kaleda/notes/report-fuzzing/

import contextlib
import errno
import functools
import json
import logging
import os
import psutil
import re
import shutil
import six
import socket
import threading
import time
import zipfile
from datetime import timedelta

import sandbox.projects.sandbox.resources as sb_resources
from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import misc as ctm
from sandbox.common.types import notification as ctn
from sandbox.common.types import task as ctt
from sandbox.projects.common import constants as consts
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.constants import constants as sdk_constants
from sandbox.projects.common.vcs import aapi
from sandbox.projects.market.report.common import helpers
from sandbox.projects.market.report.MarketReportCdRelease import MarketReportCdRelease
from sandbox.projects.security.ReportFuzzing.common import \
    create_dir, export_into_zip, fix_arc_path, get_zip_resource, drop_lite_deps, chmod_exec, chmod_pub, \
    LITE_DEPS_MAPPING, SHARED_DIR, rm_tags, add_tag, REVISION_TAG, CORPUS_TAG, INDEX_TAG, \
    find_file, get_latest_resource
from sandbox.projects.security.ReportFuzzing.resources import ReportCollectedIndex, ReportCollectedCorpus, \
    ReportFuzzExecutable, ReportFiredCorpus, ReportFuzzTreasure, ReportCollectedLiteIndex
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.environments import PipEnvironment


# Task requirements:
FOR_SYS_RAM = 2048  # Mb
THREAD_RAM = 8192  # Mb
DISK_SPACE = 20  # Gb
THREADS = 2  # Mb
# Report config:
HFND_TCP_PORT = 18917

# Filesystem config:
WORK_DIR = os.getenv('TASK_DIR', '.')

RE_FINDINGS = re.compile(r'(\w+(?:-\w+)*)-([\d\w]{40})')
RE_LOGS = re.compile(r'fuzz-\d+\.log')
RE_CORE_DUMPS = re.compile(r'fuzz_driver\.\d+\..+')
LITE_INDEX_CONST = 'Lite'
HUGE_INDEX_CONST = 'Huge'

logger = logging.getLogger(__name__)
MAX_MOUNT_RETRIES = 5
MOUNT_RETRY_DELAY = 60  # seconds

RE_CONF_PORT = re.compile(r'\bPort:?\s+(\d+)')

SHADE_START_TIMEOUT = 60
BASE_REPORT_START_TIMEOUT = 600
FUZZ_DRIVER_START_TIMEOUT = 1800


def add_write_permission(path):
    cmd = ('chmod', '-R', 'u+w', path)
    handler = process.run_process(cmd, wait=True, check=True)
    handler.wait()


def is_port_open(port):
    if port is None:
        return True
    with contextlib.closing(socket.socket(socket.AF_INET)) as sock:
        sock.settimeout(2)
        return sock.connect_ex(('127.0.0.1', port)) == 0


def wait_process_run(handler, name, port, timeout):
    logger.info('Wait while %s starting at port %s', name, port)
    begin = time.time()
    while True:
        if not psutil.pid_exists(handler.pid):
            raise RuntimeError('Process {} terminated unexpectedly'.format(name))
        if is_port_open(port):
            break
        now = time.time()
        if now > begin + timeout:
            raise RuntimeError('Timeout {}s was reached while waiting start of {}'.format(timeout, name))
        time.sleep(1)
    logger.info('%s started successfully', name)


def make_start_waiter(*args):
    return functools.partial(wait_process_run, *args)


def get_service_port(cfg_path):
    with open(cfg_path) as cfg:
        for line in cfg:
            mo = RE_CONF_PORT.search(line)
            if mo:
                return int(mo.group(1))
    return None


class ReportBuildAndFuzz(sdk2.Task):
    _ya_build_params = {
        consts.SANITIZE: 'address',
        consts.SANITIZER_FLAGS: ['-fsanitize=fuzzer'],
        consts.DOWNLOAD_ARTIFACTS_FROM_DISTBUILD: True,
        consts.FORCE_BUILD_DEPENDS: True,
    }
    _ya_build_def_flags = {
        'FORCE_VCS_INFO_UPDATE': 'yes',
        'FUZZING': 'yes',
        'CFLAGS': '-D_HF_ARCH_LINUX',
    }
    _treasures_archive_name = 'treasures.zip'
    _logs_corpus_filename = 'logs_corpus.zip'
    _corpus_dir = os.path.join(WORK_DIR, 'corpus')
    _new_corpus_dir = os.path.join(WORK_DIR, 'new_corpus')
    _build_dir = os.path.join(WORK_DIR, 'build')

    _collecting_results_lock = threading.Lock()
    _collect_was_run = False
    _treasure_push_lock = threading.Lock()
    _emergency_treasure_push = False

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Building') as build_group:
            arc_yav_token = sdk2.parameters.YavSecret('Yav ARC_TOKEN')
            yt_yav_token = sdk2.parameters.YavSecret('Yav YT_STORE_TOKEN')
            repo_path = sdk2.parameters.String(
                'Arcadia repo path',
                required=True,
                default_value='market/report/report_meta'
            )
            exec_name = sdk2.parameters.String(
                'Target executable name',
                default='fuzz_driver'
            )
            revision = sdk2.parameters.Integer(
                'Revision (0 means latest)',
                default_value=0,
                hint=True
            )
            with sdk2.parameters.RadioGroup('Build type') as build_type:
                build_type.values[consts.RELEASE_BUILD_TYPE] = build_type.Value('Release', default=True)
                build_type.values[consts.DEBUG_BUILD_TYPE] = build_type.Value('Debug')
                build_type.values[consts.RELEASE_WITH_DEBUG_INFO_BUILD_TYPE] = build_type.Value(
                    'Release with debug info')

            report_exec_res = sdk2.parameters.Resource(
                'Get report executable from resource (cancels build if specified). '
                'Remember that executable should be compiled with sanitiser flags.'
            )
            drop_deps = sdk2.parameters.Bool(
                'Drop lite deps',
                default_value=False
            )
            run_shade = sdk2.parameters.Bool(
                'Run shade',
                default_value=False
            )
            use_stable_base = sdk2.parameters.Bool(
                'Use stable base report',
                default_value=False
            )
        with sdk2.parameters.Group('Index & Corpus') as index_corpus_group:
            with sdk2.parameters.RadioGroup('Index', required=True) as index_type:
                index_type.values[LITE_INDEX_CONST] = index_type.Value('Lite index', default=True)
                index_type.values[HUGE_INDEX_CONST] = index_type.Value('Huge index')

        with sdk2.parameters.Group('Fuzzing') as fuzzing_group:
            build_only = sdk2.parameters.Bool(
                'Disable fuzzing. Only build fuzz driver executable',
                default_value=True
            )
            jobs = sdk2.parameters.Integer(
                'jobs: Flag indicates that that N fuzzing jobs should be run to completion '
                '(i.e. until a bug is found or time/iteration limits are reached). '
                'These jobs will be run across a set of worker processes, '
                'by default using half of the available CPU cores; the count of worker processes can be overridden '
                'by the -workers=N option. For example, running with -jobs=30 on a 12-core machine would run 6 '
                'workers by default, with each worker averaging 5 bugs by completion of the entire process. ',
                default_value=10000
            )
            workers = sdk2.parameters.Integer(
                'workers: Number of simultaneous worker processes to run the fuzzing jobs to completion in. '
                'If 0 (the default), min(jobs, NumberOfCpuCores()/2) is used.',
                default_value=THREADS
            )
            rss_limit = sdk2.parameters.Integer(
                'rss_limit_mb: Memory usage limit in Mb, default 24576. Use 0 to disable the limit. '
                'If an input requires more than this amount of RSS memory to execute, '
                'the process is treated as a failure case. '
                'The limit is checked in a separate thread every second.',
                default_value=24576
            )
            max_len = sdk2.parameters.Integer(
                'max_len: Maximum length of a test input. If 0 (the default), '
                'libFuzzer tries to guess a good value based on the corpus.',
                default_value=512
            )
            alloc_may_return_null = sdk2.parameters.Bool(
                'ASan option: Allow allocator to return null',
                default_value=False
            )
            detect_leaks = sdk2.parameters.Bool(
                'Detect leaks',
                default_value=True
            )
            fuzz_timeout = sdk2.parameters.Integer(
                'Timeout in seconds. If an input takes longer than this timeout, '
                'the process is treated as a failure case.',
                default_value=300
            )
            max_total_fuzz_time = sdk2.parameters.Integer(
                'If positive, indicates the maximum total time in seconds to run the fuzzer. '
                'If 0 (the default), run indefinitely.',
                default_value=0
            )
        with sdk2.parameters.Output():
            with sdk2.parameters.Group('Output') as results_group:
                findings = sdk2.parameters.JSON('Findings')

        _container = sdk2.parameters.Container(
            'Report fuzzing environment',
            resource_type=sb_resources.LXC_CONTAINER,
            attrs=dict(target='report-fuzzing-env'),
            required=True
        )

        notifications = [
            sdk2.Notification(
                [ctt.Status.FAILURE, ctt.Status.EXCEPTION],
                ['host=market_report_fuzzing&service=run_fuzzing'],
                ctn.Transport.JUGGLER,
                check_status=ctn.JugglerStatus.CRIT
            ),
            sdk2.Notification(
                [ctt.Status.TIMEOUT, ctt.Status.STOPPED],
                ['host=market_report_fuzzing&service=run_fuzzing'],
                ctn.Transport.JUGGLER,
                check_status=ctn.JugglerStatus.OK
            )
        ]

    class Requirements(sdk2.Task.Requirements):
        disk_space = DISK_SPACE * 1024
        ram = FOR_SYS_RAM + THREAD_RAM * THREADS
        cores = THREADS
        environments = [
            # Building:
            PipEnvironment('wheel', version='0.30.0'),
        ]
        dns = ctm.DnsType.DNS64

    def build_and_fuzz(self, target_platform='linux'):
        mount_args = dict()
        token = self._get_token(self.Parameters.arc_yav_token, vault_item='ARC_TOKEN')
        if token:
            os.environ.update({'ARC_TOKEN': token})
            mount_args.update({'arc_oauth_token': token})
        with arcadiasdk.mount_arc_path(
                self._get_arc_url(),
                **mount_args
        ) as arc_root:
            # DROPPING/COMPILING LITE DEPS
            if self.Parameters.drop_deps:
                drop_lite_deps(arc_root)
            else:
                arcadiasdk.do_build(
                    build_system=sdk_constants.DISTBUILD_BUILD_SYSTEM,
                    build_type=self.Parameters.build_type,
                    add_result=['pb2.py', '.a', '.h'],
                    results_dir=ReportBuildAndFuzz._build_dir,
                    source_root=arc_root,
                    targets=[os.path.join(arc_root, 'market', 'report')],
                    target_platform=target_platform,
                    clear_build=False,
                )

            # GET STABLE BASE REPORT
            if self.Parameters.use_stable_base:
                base_report_dir = os.path.abspath(os.path.join(WORK_DIR, 'base_report'))
                if not os.path.exists(base_report_dir):
                    try:
                        os.makedirs(base_report_dir)
                    except OSError as e:
                        if e.errno != errno.EEXIST:
                            raise
                task = helpers.find_last_task(MarketReportCdRelease.type, status=ctt.Status.Group.SUCCEED)
                resource = helpers.find_last_report_resourse(release_task=task)
                self.base_report = helpers.unpack_report_bin(str(base_report_dir), resource=resource)
            else:
                self.base_report = None

            # REPORT BUILDING
            if self.Parameters.report_exec_res:
                self.fuzz_driver = os.path.abspath(os.path.join(WORK_DIR, str(self.Parameters.exec_name)))
                with open(self.fuzz_driver, 'wb') as exe:
                    exe.write(sdk2.ResourceData(self.Parameters.report_exec_res).path.read_bytes())
            else:
                self.fuzz_driver, self.build_meta = self.fuzz_build(
                    arc_root,
                    target=fix_arc_path(str(self.Parameters.repo_path)),
                    build_type=self.Parameters.build_type
                )
                self.export_executable(self.fuzz_driver, self.build_meta)

            if self.Parameters.build_only:
                return

            # INDEX AND CORPUS COLLECTION
            corpus, res_id = get_zip_resource('corpus', ReportCollectedCorpus)
            add_tag(self, CORPUS_TAG.format(res_id))
            logger.info('Fetched corpus id={} items_count={}'.format(res_id, len(os.listdir(corpus))))
            self._corpus_id = res_id
            if os.path.isdir(SHARED_DIR):
                shutil.rmtree(SHARED_DIR)
            create_dir(SHARED_DIR)
            chmod_pub(SHARED_DIR)

            if self.Parameters.index_type == LITE_INDEX_CONST:
                res = get_latest_resource(ReportCollectedLiteIndex)
                res_id = res.id
                res_data = sdk2.ResourceData(res)
                index = str(res_data.path)
                logger.info('Fetched lite index id {} into {}: {}'
                            .format(res_id, index, os.listdir(index)))
                index = os.path.join(index, 'test_prime')
            elif self.Parameters.index_type == HUGE_INDEX_CONST:
                index, res_id = get_zip_resource(
                    os.path.join(SHARED_DIR, 'index'),
                    ReportCollectedIndex,
                )
                add_tag(self, INDEX_TAG.format(res_id))
                logger.info('Fetched huge index id {} into {}: {}'.format(res_id, index, os.listdir(index)))
            else:
                raise Exception('Unknown index type {}'.format(self.Parameters.index_type))

            self._index_id = res_id
            add_write_permission(index)

            if self.Parameters.run_shade:
                if self.Parameters.drop_deps:
                    shade = os.path.join(arc_root, LITE_DEPS_MAPPING['shade'])
                else:
                    shade = os.path.join(arc_root, 'market', 'shade', 'bin', 'bin')
                    if not os.path.isfile(shade):
                        arcadiasdk.do_build(
                            build_system=consts.YMAKE_BUILD_SYSTEM,
                            build_type=consts.RELEASE_BUILD_TYPE,
                            results_dir=ReportBuildAndFuzz._build_dir,
                            source_root=arc_root,
                            targets=[os.path.join(arc_root, 'market', 'shade')],
                            clear_build=False
                        )
                        shade = find_file(ReportBuildAndFuzz._build_dir, 'bin')
                proc_shade = self.start_shade(shade, index)
                with self._procs_lock:
                    self._active_procs.append(proc_shade)

            if self.base_report:
                base_report_config = os.path.join(index, 'report_base', 'config', 'report_config.conf')
                if not os.path.isfile(base_report_config):
                    raise Exception('Base report config not found in {} ({}). Check your index.'.format(
                        base_report_config, os.listdir(os.path.dirname(base_report_config))
                    ))
                proc_base_report = self.start_base_report(
                    self.base_report,
                    base_report_config,
                    index
                )
                with self._procs_lock:
                    self._active_procs.append(proc_base_report)

            report_config = os.path.join(index, 'report_meta', 'config', 'report_config.conf')
            if not os.path.isfile(report_config):
                raise Exception('Report config not found in {} ({}). Check your index.'.format(
                    report_config, os.listdir(os.path.dirname(report_config))
                ))

            logger.info('Wait all test environment processes')
            with self._procs_lock:
                for _, wait_start in self._active_procs:
                    wait_start()

            # FUZZING
            logger.info('Start fuzzing')
            create_dir(self._new_corpus_dir)
            try:
                proc_handler = self.start_fuzzing(
                    fuzz_bin_path=self.fuzz_driver,
                    report_conf=report_config,
                    new_corpus=self._new_corpus_dir,
                    corpus=self._corpus_dir,
                    rss_limit=self.Parameters.rss_limit,
                    max_len=self.Parameters.max_len,
                    jobs=self.Parameters.jobs,
                    workers=self.Parameters.workers,
                    detect_leaks=self.Parameters.detect_leaks,
                    timeout=self.Parameters.fuzz_timeout,
                    max_total_time=self.Parameters.max_total_fuzz_time,
                    alloc_may_ret_null=self.Parameters.alloc_may_return_null
                )
                with self._procs_lock:
                    self._active_procs.append(proc_handler)
                proc_handler[0].wait()  # wait process finish
            finally:
                self.collect_fuzz_findings(WORK_DIR)

    # LIFE CYCLE CALLBACKS #

    def on_execute(self):
        if self.Parameters.revision == 0:
            self._revision = aapi.ArcadiaApi.svn_head()
        else:
            self._revision = self.Parameters.revision
        if self.Parameters.report_exec_res is not None and \
                hasattr(self.Parameters.report_exec_res, 'revision'):
            self._revision = self.Parameters.report_exec_res.revision

        if isinstance(self._revision, six.string_types) and not self._revision.isdigit():
            self.set_info('Fuzzing does not work with hotfix branches')
            return

        self._procs_lock = threading.Lock()
        self._active_procs = []     # pairs of proc handler and start waiter
        rm_tags(self)
        add_tag(self, REVISION_TAG.format(self._revision))
        for _ in range(MAX_MOUNT_RETRIES):
            try:
                self.build_and_fuzz()
                logger.info('Task finished')
                return  # arc mount succeeded
            except AssertionError as e:
                logger.warning(e)
                time.sleep(MOUNT_RETRY_DELAY)
                logger.info('Retrying arc mount one more time')
        raise Exception('Unable to mount arc r{} after {} retries (retry delay = {})'.format(
            self._revision, MAX_MOUNT_RETRIES, MOUNT_RETRY_DELAY)
        )

    def timeout_checkpoints(self):
        return [10, 30, 60, 60 * 3, 60 * 5]

    def on_before_timeout(self, seconds):
        logger.info('on_before_timeout({})'.format(seconds))
        if seconds <= 30:
            with ReportBuildAndFuzz._treasure_push_lock:
                logger.warning('Set treasure push flag')
                ReportBuildAndFuzz._emergency_treasure_push = True
        if seconds <= 60 * 5:
            self.collect_fuzz_findings(WORK_DIR)

    def on_terminate(self):
        logger.info('SIGTERM: terminating signal received')
        self.collect_fuzz_findings(WORK_DIR)

    # MAIN METHODS #

    def fuzz_build(
            self,
            arc_root,
            target,
            build_type,
            target_platform='linux',
            clear_build=False
    ):
        """
        manual CL command:
         `ya make -DFUZZING -DCFLAGS=-D_HF_ARCH_LINUX -r --force-build-depends --add-result pb2.py
         --add-result .a --add-result .h --sanitize=address --sanitizer-flag=-fsanitize=fuzzer
         --sanitize-coverage=trace-div,trace-gep --dist -E`
        :return: path to executable and build meta info
        """
        build_system = sdk_constants.SEMI_DISTBUILD_BUILD_SYSTEM
        yt_token = self._get_token(self.Parameters.yt_yav_token, vault_item='YT_STORE_TOKEN')
        arcadiasdk.do_build(
            build_system=build_system,
            build_type=build_type,
            add_result=['pb2.py', '.a', '.h'],
            results_dir=ReportBuildAndFuzz._build_dir,
            source_root=arc_root,
            targets=[target],
            target_platform=target_platform,
            clear_build=clear_build,
            def_flags=ReportBuildAndFuzz._ya_build_def_flags,
            yt_store_params=arcadiasdk.YtStoreParams(True, yt_token),
            **ReportBuildAndFuzz._ya_build_params
        )
        meta = dict(
            env=build_type,
            params=ReportBuildAndFuzz._ya_build_params,
            def_flags=ReportBuildAndFuzz._ya_build_def_flags,
            clear_build=clear_build,
            target_platform=target_platform,
            build_system=build_system
        )
        return find_file(ReportBuildAndFuzz._build_dir, self.Parameters.exec_name), meta

    def _get_arc_url(self):
        return 'arcadia-arc:/#r{}'.format(self._revision)

    def start_shade(self, shade, index, workdir=WORK_DIR):
        shade_conf = os.path.join(index, 'shade', 'conf', 'server')

        cmd = (shade, '-c', shade_conf)
        with open('shade_log.txt', 'w') as shade_log:
            handler = process.run_process(
                cmd, log_prefix='shade_logs', work_dir=workdir,
                outputs_to_one_file=False, stdout=shade_log,
                wait=False, check=False
            )
        waiter = make_start_waiter(handler, 'shade', get_service_port(shade_conf), SHADE_START_TIMEOUT)
        return (handler, waiter)

    def start_base_report(self, report_bin_path, report_conf, index, workdir=WORK_DIR):
        cmd = (report_bin_path, '-d', report_conf)
        env = {
            'REPORT_CONFIG': report_conf
        }
        with open('base_report_log.txt', 'w') as base_report_log:
            handler = process.run_process(
                cmd, log_prefix='base_report_logs', work_dir=workdir,
                outputs_to_one_file=False, stdout=base_report_log,
                environment=env, wait=False, check=False, shell=True
            )
        waiter = make_start_waiter(handler, 'base_report', get_service_port(report_conf), BASE_REPORT_START_TIMEOUT)
        return (handler, waiter)

    def start_fuzzing(
            self,
            fuzz_bin_path,
            report_conf,
            new_corpus,
            corpus,
            rss_limit,
            max_len,
            jobs,
            workers,
            workdir=WORK_DIR,
            detect_leaks=True,
            timeout=1200,
            max_total_time=0,
            alloc_may_ret_null=True
    ):
        """
        Fuzzing
        :param fuzz_bin_path: arcadia/market/report/report_bin/fuzz_driver
        :param report_conf: /tmp/fuzzing-shared/index/config/report.conf
        :param new_corpus: /tmp/fuzzing-shared/new_corpus
        :param corpus: /tmp/fuzzing-shared/corpus
        :param rss_limit: 24576
        :param max_len: 512
        :param jobs: 2
        :param workers: 10000
        :param workdir:
        :param detect_leaks: True
        :param timeout:
        :param max_total_time:
        :param alloc_may_ret_null: False
        """
        chmod_exec(fuzz_bin_path)
        args = [
            '-rss_limit_mb={}'.format(rss_limit),
            '-max_len={}'.format(max_len),
            '-jobs={}'.format(jobs),
            '-workers={}'.format(workers),
            '-detect_leaks={}'.format(int(detect_leaks)),
            '-print_final_stats=1',
            '-timeout={}'.format(timeout),
            '-max_total_time={}'.format(max_total_time),
            new_corpus,
            corpus
        ]
        env = {
            'HFND_TCP_PORT': HFND_TCP_PORT,
            'HFND_WAIT_SERVER_TIMEOUT': FUZZ_DRIVER_START_TIMEOUT,
            'PROG_ARGUMENTS': '-d {} -p {}'.format(report_conf, HFND_TCP_PORT),
            'REPORT_CONFIG': report_conf,
        }
        asan_opts = []
        if alloc_may_ret_null:
            asan_opts.append('allocator_may_return_null=1')
        if len(asan_opts) > 0:
            env.update({
                'ASAN_OPTIONS': ':'.join(asan_opts)
            })
        #  /usr/bin/unshare -Ur ./fuzz_driver ... /tmp/fuzzing-shared/new_corpus /tmp/fuzzing-shared/corpus
        cmd = ['/usr/bin/unshare', '-Ur', fuzz_bin_path] + args
        logger.info('Fuzzing command: {} Env: {}'.format(str(cmd), str(env)))
        self._fuzz_start_time = time.time()
        handler = process.run_process(
            cmd, log_prefix='fuzz_logs', work_dir=workdir,
            outputs_to_one_file=False, environment=env,
            wait=False, check=False
        )
        return (handler, lambda: None)

    @staticmethod
    def _cleanup():
        logger.info('Removing build files...')
        build_dir = ReportBuildAndFuzz._build_dir
        if os.path.isdir(build_dir) and not os.path.islink(build_dir):
            shutil.rmtree(build_dir)
        elif os.path.exists(build_dir):
            os.remove(build_dir)
        logger.info('Build files removed')

    def collect_fuzz_findings(self, where, workdir=WORK_DIR):
        with ReportBuildAndFuzz._collecting_results_lock:  # Allowed to call only once
            if ReportBuildAndFuzz._collect_was_run:
                return
            ReportBuildAndFuzz._collect_was_run = True
        with self._procs_lock:
            logger.info('Killing processes')
            for handler, _ in self._active_procs:
                try:
                    handler.kill()
                except OSError as e:
                    logger.warning('Unable to kill: {}'.format(e))
        ReportBuildAndFuzz._cleanup()

        fuzz_runtime = str(timedelta(seconds=time.time() - self._fuzz_start_time))
        files = os.listdir(where)
        logger.info('Total items in dir: {}'.format(len(os.listdir(where))))
        stats = dict()
        zipname = ReportBuildAndFuzz._treasures_archive_name
        zippath = os.path.join(workdir, zipname)
        with zipfile.ZipFile(zippath, 'w', zipfile.ZIP_DEFLATED) as findings:
            for f in files:
                with ReportBuildAndFuzz._treasure_push_lock:
                    if ReportBuildAndFuzz._emergency_treasure_push:
                        logger.warning('Unable to process all the items! Treasure is truncated!')
                        break

                m = re.match(RE_FINDINGS, f)
                if m:
                    filename = os.path.join(workdir, m.group(0))
                    vuln_type = m.group(1)
                    stats.update({vuln_type: stats.get(vuln_type, 0) + 1})
                    logger.info('Exporting finding "{}"'.format(filename))
                    findings.write(filename)

                m = re.match(RE_LOGS, f)
                if m:
                    filename = os.path.join(workdir, m.group(0))
                    logger.info('Exporting log item "{}"'.format(filename))
                    findings.write(filename)
                m = re.match(RE_CORE_DUMPS, f)
                if m:
                    logger.info('Found core dump file "{}"'.format(m.group(0)))
        logger.info('Zip with treasures prepared')
        self.Parameters.findings = json.dumps(stats)

        if len(stats) == 0:
            logger.info('No treasures found for export. Exporting logs only')
            # TODO: Set SUCCESS status to the task
        self.export_treasure(zipname, stats, fuzz_runtime)

        self.collect_new_corpus(ReportBuildAndFuzz._new_corpus_dir)
        logger.info('Fuzz findings collected successfully')

    def collect_new_corpus(self, new_corpus_path):
        archive, n = export_into_zip(new_corpus_path, 'new_corpus.zip')
        if n > 0:
            self.export_new_corpus(archive, n)

    def export_new_corpus(self, filename, items_count):
        description = 'Found {} interesting items'.format(items_count)
        sdk2.ResourceData(ReportFiredCorpus(
            self, description, filename,
            revision=self._revision,
            items_count=items_count
        )).ready()

    def export_treasure(self, filename, stats, runtime):
        description = 'Fuzz runtime:\t{}\n' \
                      'Treasures collected:\t{}'.format(
            runtime, sum(stats.values())
        )
        sdk2.ResourceData(ReportFuzzTreasure(
            self, description, filename,
            revision=self._revision,
            fuzz_runtime=runtime,
            findings=json.dumps(stats),
            index_id=self._index_id,
            corpus_id=self._corpus_id
        )).ready()

    def export_executable(self, filename, meta_info):
        sdk2.ResourceData(ReportFuzzExecutable(
            self, str(meta_info), filename,
            revision=self._revision,
            build_type=meta_info.get('env', 'unknown')
        )).ready()

    def _get_token(self, yav_token, vault_item=None):
        try:
            if yav_token:
                return yav_token.data()[yav_token.default_key]
            return sdk2.Vault.data(self.owner, vault_item) if vault_item else None
        except Exception as e:
            logger.exception(e)
            raise errors.TaskFailure('Failed to get token: {}'.format(e))
