import logging
import os
import re
import shutil
import stat
import tarfile
import zipfile

from sandbox import sdk2
from sandbox.projects.security.ReportFuzzing.resources import MARKET_REPORT_LITE_DEPENDS

logger = logging.getLogger(__name__)
WORK_DIR = '.'
MOUNTED_NAMESPACE = '/tmp/HFND_TMP_DIR'
SHARED_DIR = '/tmp/fuzzing-shared'

REVISION_TAG = 'REVISION-{}'
REVISION_RE = re.compile(REVISION_TAG.format(r'(\d+)'))
MARKET_REPORT_TAG = 'MARKET_REPORT-{}'
MARKET_REPORT_RE = re.compile(MARKET_REPORT_TAG.format(r'(\d+)'))
CORPUS_TAG = 'CORPUS-{}'
CORPUS_RE = re.compile(CORPUS_TAG.format(r'(\d+)'))
INDEX_TAG = 'INDEX-{}'
INDEX_RE = re.compile(INDEX_TAG.format(r'(\d+)'))
PROCESSED_TAG = 'PROCESSED'
PROCESSED_RE = re.compile(INDEX_TAG.format(r'(\d+)'))

# File system mappings
LITE_DEPS_MAPPING = {
    'balancer': 'balancer/daemons/balancer/balancer',
    'lbk_recipe': 'kikimr/public/tools/lbk_recipe/lbk_recipe',
    'indexarc-converter': 'market/idx/generation/indexarc-converter/indexarc-converter',
    'currency_pack.so': 'idx/quick/pylibrary/currency_pack/so/currency_pack.so',
    'pbsncat': 'market/idx/tools/pbsncat/bin/pbsncat',
    'quoter': 'market/quoter/bin/bin',
    'shade': 'market/shade/bin/bin',
    'indexann_generator': 'market/tools/indexann_generator/indexann_generator',
    'indexerf_generator': 'market/tools/indexerf_generator/indexerf_generator',
    'jump-table-dumper': 'market/tools/jump_table_dumper/bin/local_dumper/jump-table-dumper',
    'microindexer': 'market/tools/microindexer/microindexer',
    'promo-secondary-offers-converter': 'market/tools/promo_secondary_offers_converter/bin/promo-secondary-offers-converter',
    'shop-vendor-promo-clicks-stats-converter': 'market/tools/shop-vendor-promo-clicks-stats-converter/shop-vendor-promo-clicks-stats-converter',
    'wizard-model-reviews-json2mmap-converter': 'market/tools/wizard-model-reviews-json2mmap-converter/wizard-model-reviews-json2mmap-converter',
    'idx_convert': 'search/panther/tools/idx_convert/idx_convert',
    'app_host_converter': 'apphost/tools/converter/converter',
    'grattrview': 'yweb/robot/tools/grattrview/grattrview'
}


def create_dir(*f_paths):
    f_path = os.path.join(*f_paths)
    if not os.path.exists(f_path):
        os.makedirs(f_path)
    return f_path


def recreate_dir(*f_paths):
    f_path = os.path.join(*f_paths)
    if not os.path.exists(f_path):
        os.makedirs(f_path)
    else:
        shutil.rmtree(f_path)
        os.makedirs(f_path)


def add_tag(context, tag):
    try:
        task_tags = context.server.task[context.id].read()["tags"]
        task_tags.append(tag)
        context.server.task[context.id].update({"tags": task_tags})
        return True
    except Exception:
        return False


def get_tags(context):
    try:
        return context.server.task[context.id].read()["tags"]
    except Exception:
        logger.error('Unable to read tags')
    return []


def rm_tags(context):
    try:
        context.server.task[context.id].update({"tags": []})
        return True
    except Exception:
        return False


def chmod_exec(file):
    st = os.stat(file)
    os.chmod(file, st.st_mode | stat.S_IEXEC)


def chmod_pub(file):
    st = os.stat(file)
    os.chmod(file, st.st_mode | stat.S_IROTH | stat.S_IWOTH)


def drop_lite_deps(arc_root, work_dir=WORK_DIR, bin_deps_folder='bin', revision=None):
    logger.info('Dropping lite deps ...')
    if revision is not None:
        tar_fpath, res_id = get_latest_resource_path(MARKET_REPORT_LITE_DEPENDS, attrs={'svn_revision': revision})
    else:
        tar_fpath, res_id = get_latest_resource_path(MARKET_REPORT_LITE_DEPENDS, attrs={'svn_revision': revision})
    logger.info('Extracting tarfile {}'.format(tar_fpath))
    with tarfile.open(tar_fpath) as tar:
        tar.extractall(path=work_dir)
    extracted_bind = os.path.join(work_dir, bin_deps_folder)
    logger.info('extracted > {}'.format(os.listdir(extracted_bind)))
    for f_from, f_to in LITE_DEPS_MAPPING.items():
        src = os.path.join(extracted_bind, f_from)
        dst = os.path.join(arc_root, f_to)
        create_dir(os.path.dirname(dst))
        shutil.copy(src, dst)
        chmod_exec(dst)
    logger.info('Dropping lite deps DONE')


def export_into_zip(filepath, zip_name, workdir=WORK_DIR):
    filepath = os.path.abspath(filepath)
    archive_filepath = os.path.abspath(os.path.join(workdir, zip_name))
    logger.info('exporting {} as {}'.format(filepath, archive_filepath))
    with zipfile.ZipFile(archive_filepath, 'w', zipfile.ZIP_DEFLATED) as f:
        n = 0
        init_prefix = os.path.abspath(os.path.expanduser(os.path.expandvars(filepath)))
        for folderName, subfolders, filenames in os.walk(filepath):
            for filename in filenames:
                saved_from = os.path.join(folderName, filename)
                saved_to = os.path.join(folderName[len(init_prefix):], filename)
                f.write(saved_from, saved_to)
                n += 1
    return archive_filepath, n


def get_zip_resource(extract_path, res_type, **extra_attrs):
    extract_path = os.path.abspath(extract_path)
    create_dir(extract_path)
    res_archive, res_id = get_latest_resource_path(res_type, **extra_attrs)
    with zipfile.ZipFile(res_archive) as zf:
        zf.extractall(extract_path)
    return extract_path, res_id


def find_file(search_path, filename):
    """
    Warning! May throw exception
    :param search_path: path to search in
    :param filename: name of a target binary
    :return: path to compiled binary
    """
    for root, _, files in os.walk(search_path):
        for file in files:
            if len(filename) > 0:
                if filename == file:
                    f_path = os.path.join(root, file)
                    logger.info('Found binary {}'.format(f_path))
                    return f_path
    raise Exception('Unable to locate file "{}" in {}'.format(filename, search_path))


def fix_arc_path(path):
    rm_scope = ('/', 'arc', 'arcadia')
    changed = True
    while changed:
        changed = False
        for rm_item in rm_scope:
            if path.startswith(rm_item):
                path = path[len(rm_item):]
                changed = True
    return path


def patch_file(file, regexp_str, replace_str):
    with open(file) as f:
        data = ''
        regexp = re.compile(regexp_str)
        for line in f:
            data += regexp.sub(
                replace_str,
                line
            )
    with open(file, 'w') as f:
        f.write(data)


def get_latest_resource(res_type, **params):
    if type(res_type) == str:
        resource = sdk2.Resource.find(
            type=res_type, state='READY', **params
        ).order(-sdk2.Resource.id).first()
    else:
        resource = res_type.find(
            state='READY', **params
        ).order(-sdk2.Resource.id).first()
    if resource is None:
        return None
    return resource


def get_latest_resource_path(res_type, **params):
    res = get_latest_resource(res_type, **params)
    return str(sdk2.ResourceData(res).path), res.id
