import json
import logging
import time
from collections import deque
from urlparse import urljoin

import requests
from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.security.ReportFuzzing.ReportBuildAndFuzz import ReportBuildAndFuzz
from sandbox.projects.security.ReportFuzzing.common import get_latest_resource, add_tag, get_tags, PROCESSED_TAG, \
    REVISION_TAG
from sandbox.projects.security.ReportFuzzing.resources import ReportFuzzTreasure, ReportFuzzExecutable, \
    ReportCollectedCorpus, ReportCollectedIndex, ReportCollectedLiteIndex

MAX_RPS = 50
MAX_RETRIES = 3
LOAD_RES_RETRY_DELAY = 30  # seconds
FUZZ_FINDINGS_TASKS_LIMIT = 10
TASK_CHECK_UPDATE_INTERVAL = 30.0
reqs = deque()

STARTREK_URL = 'https://st-api.yandex-team.ru/v2'
STARTREK_TESTING_URL = 'https://st-api.test.yandex-team.ru/v2'  # For tests
ARC_URL_REV = 'https://a.yandex-team.ru/arc/trunk/arcadia?rev=r{}'
SB_RES_ID_URL = 'https://sandbox.yandex-team.ru/resource/{}/view'
DEFAULT_QUEUE = 'REPORTFUZZING'
WIKI_TREASURE_ANALYSIS = 'https://wiki.yandex-team.ru/users/ya-andrei/fuzztreasureanalysis/'
WIKI_FUZZING_NOTES = 'https://wiki.yandex-team.ru/security/internal/service-security/market/report-fuzzing'

logger = logging.getLogger(__name__)


class ReportCollectTreasures(sdk2.Task):
    class Parameters(sdk2.Parameters):
        revision = sdk2.parameters.Integer(
            'Revision (0 means fetch latest task)',
            default_value=0,
            hint=True
        )
        startreck_token = sdk2.parameters.String(
            'Sandbox vault STARTRECK_TOKEN name [deprecated]',
            default_value='STARTRECK_TOKEN',
            required=True
        )
        startreck_yav_token = sdk2.parameters.YavSecret(
            'Yav STARTRECK_TOKEN'
        )
        startreck_yav_token_key = sdk2.parameters.String(
            'Yav STARTRECK_TOKEN key',
            default_value='STARTRECK_TOKEN'
        )
        timeout = sdk2.parameters.Integer(
            'Wait timeout in seconds (negative value means wait forever, 0 value means do not wait)',
            default_value=3600,
            required=True
        )
        queue = sdk2.parameters.String(
            'Ticket queue',
            default_value=DEFAULT_QUEUE,
            required=True
        )
        process_last_only = sdk2.parameters.Bool(
            'if enabled looks only at the last fuzz run. '
            'Otherwise takes {} latest with the revision provided'.format(FUZZ_FINDINGS_TASKS_LIMIT),
            default_value=True
        )
        testing = sdk2.parameters.Bool(
            'Create tickets on st-api.test',
            default_value=False
        )
        ignore_list = sdk2.parameters.String(
            'Comma separated names of treasures to ignore',
            default_value='slow-unit,'
        )
        with sdk2.parameters.Output():
            with sdk2.parameters.Group('Output') as results:
                findings = sdk2.parameters.JSON('Findings')
                ticket = sdk2.parameters.String('Ticket')
                check_passed = sdk2.parameters.Bool('Check passed', default_value=False)

    def init_params(self):
        Ticket.STARTREK_URL = STARTREK_URL
        if self.Parameters.testing:
            Ticket.STARTREK_URL = STARTREK_TESTING_URL

        Ticket.QUEUE = str(self.Parameters.queue)

    @staticmethod
    def _filter_func(processed_tag=True, no_resource=True, no_findings=True):
        def _filter(task):
            if processed_tag and (PROCESSED_TAG in get_tags(task)):
                return False
            res = get_latest_resource(ReportFuzzTreasure, task_id=task.id)
            if no_resource and res is None:
                return False
            if no_findings and len(json.loads(res.findings)) == 0:
                return False
            return True

        return _filter

    def on_execute(self):
        self.init_params()
        start_time = time.time()

        first_run = True
        while time.time() - start_time <= self.Parameters.timeout or first_run:
            first_run = False
            kwargs = dict()
            if self.Parameters.revision != 0:
                kwargs.update({'tags': [REVISION_TAG.format(self.Parameters.revision)]})
            tasks = list(ReportBuildAndFuzz.find(
                status=ctt.Status.Group.FINISH - {ctt.Status.DELETED} +
                       {ctt.Status.TIMEOUT, ctt.Status.STOPPED, ctt.Status.EXCEPTION},
                **kwargs
            ).order(-sdk2.Task.id).limit(FUZZ_FINDINGS_TASKS_LIMIT))
            if tasks is None or len(tasks) == 0:
                time.sleep(TASK_CHECK_UPDATE_INTERVAL)
                continue
            tasks = filter(ReportCollectTreasures._filter_func(
                no_resource=False, no_findings=False
            ), tasks)
            if len(tasks) == 0:
                raise Exception('No treasures to collect - recent fuzz tasks failed/already processed')

            if self.Parameters.process_last_only:
                tasks = tasks[:1]
            ticket = self.process_findings(
                tasks
            )
            self.Parameters.ticket = str(ticket)
            return  # Success
        raise Exception('Timeout exception')

    def _get_session(self):
        session = requests.Session()
        if self.Parameters.startreck_yav_token:
            token = self.Parameters.startreck_yav_token.data()[self.Parameters.startreck_yav_token_key]
        else:
            token = sdk2.Vault.data(self.owner, self.Parameters.startreck_token)
        session.headers.update({
            'Authorization': 'OAuth {}'.format(token)
        })
        return session

    def process_findings(self, tasks):
        logger.info('Actual fuzz task ids for revision r{} are {}'.format(
            self.Parameters.revision,
            list(map(lambda task: task.id, tasks))
        ))

        session = self._get_session()

        ticket = FuzzTicket(self, tasks)
        stats = ticket.get_issue_stats()
        logger.info('Process findings > Stats > {}'.format(stats))
        self.Parameters.findings = json.dumps(stats)

        if any(map(lambda stats: self.should_create_ticket(stats), stats.values())):
            self.Parameters.check_passed = False
            return ticket.create(session)
        else:
            self.Parameters.check_passed = True
        return ''

    def should_create_ticket(self, stats):
        copy_stats = stats.copy()
        ignore_list = map(unicode.strip, self.Parameters.ignore_list.split(','))
        for ignore in ignore_list:
            if ignore in copy_stats:
                copy_stats.pop(ignore)
        return len(copy_stats) > 0


def rps_limit(func):
    def _func(*args, **kwargs):
        global reqs
        cnt = 0
        for _ in range(len(reqs)):
            req = reqs.popleft()
            if time.time() - req < 1.0:
                reqs.append(req)
                cnt += 1
        if cnt >= MAX_RPS:
            time.sleep(1.0)
        reqs.append(time.time())
        return func(*args, **kwargs)

    return _func


class Style:
    @staticmethod
    def bold(text):
        return '**{}**'.format(text)

    @staticmethod
    def url(url, text):
        return '(({} {}))'.format(url, text)

    @staticmethod
    def li(*text_items):
        return '\n'.join(map(lambda i: '* ' + i, text_items))

    @staticmethod
    def red(text):
        return '!!{}!!'.format(text)

    @staticmethod
    def rev_url(revision):
        return Style.url(ARC_URL_REV.format(revision), 'r{}'.format(revision))

    class Table:
        def __init__(self, *row):
            self.rows = []
            if len(row) > 0:
                self.row(*row)

        def row(self, *items):
            if len(self.rows) == 0:
                items = list(map(Style.bold, items))
            row = ' | '.join(map(str, items))
            self.rows.append('|| {} ||'.format(row))
            return self

        def __str__(self):
            return '#|\n{}\n|#\n'.format('\n'.join(self.rows))


class Ticket:
    QUEUE = DEFAULT_QUEUE
    STARTREK_URL = STARTREK_URL

    def __init__(self, summary, description, queue=QUEUE):
        self._props = dict(
            queue=queue,
            summary=summary,
            description=description
        )
        self._attachments = dict()
        self._attachment_ids = []
        self._key = None

    @rps_limit
    def _upload_attachments(self, session):
        for filename, bin_data in self._attachments.items():
            resp = session.post(
                urljoin(Ticket.STARTREK_URL, 'v2/attachments?filename={}'.format(filename)),
                files={filename: bin_data}
            ).json()
            logger.info('_upload_attachments API response: {}'.format(resp))
            self._attachment_ids.append(resp['id'])

    @rps_limit
    def _create_ticket(self, session, startrek_url=STARTREK_URL):
        resp = session.post(
            urljoin(startrek_url, 'v2/issues'),
            json=self.info()
        ).json()
        return resp

    def info(self):
        if len(self._attachment_ids) > 0:
            self._props.update({'attachmentIds': self._attachment_ids})
        return self._props

    def update_description(self, description):
        self._props['description'] = description
        return self

    def create(self, session, startrek_url=STARTREK_URL):
        if self._key is not None:
            raise Exception('Ticket already created ({})'.format(self._key))
        self._upload_attachments(session)
        resp = self._create_ticket(session, startrek_url)
        try:
            self._key = resp['key']
        except Exception:
            logger.error('Failed to create ticket: {}'.format(resp))
        logger.info('Created ticket {}'.format(self._key))
        return self._key

    def add_attachment(self, filename, filepath):
        with open(filepath, 'rb') as f:
            bin_data = f.read()
        self._attachments.update({filename: bin_data})
        return self

    def add_resource_attachment(self, filename, resource):
        if resource is None:
            logger.warning('Unable to export resource under name "{}". Resource is None'.format(filename))
            return self  # Just ignore

        for _ in range(MAX_RETRIES):
            try:
                logger.info('Adding attachment res id={}'.format(resource.id))
                filepath = str(sdk2.ResourceData(resource).path)
                self.add_attachment(filename, filepath)
                break
            except Exception as e:
                logger.error('Error {} loading resource id={}'.format(
                    e,
                    resource.id
                ))
                time.sleep(LOAD_RES_RETRY_DELAY)

        logger.error('Skipped loading res id={}'.format(resource.id))
        return self

    def __str__(self):
        return json.dumps(self.info())


class BrokenTicket(object, Ticket):
    def __init__(self, sb_task, **kwargs):
        self._sb_task = sb_task
        description = self.gen_description()
        Ticket.__init__(self, 'Index & corpus collection broken', description, **kwargs)

    def gen_description(self):
        rev = self._sb_task.Parameters.revision
        task_id = self._sb_task.id
        rev_url = Style.rev_url(rev)
        sb_url = Style.url(
            'https://sandbox.yandex-team.ru/task/{}/view'.format(task_id),
            '{}'.format(task_id),
        )
        return '\n'.join((
            'Revision:\t{}'.format(rev_url),
            'SB task:\t{}'.format(sb_url)
        ))


class FuzzTicket(object, Ticket):
    def __init__(self, context, sb_tasks):
        self._context = context
        self._revisions = []
        self._sb_tasks = []
        self._stats = []
        self._fuzz_times = []
        self._index = []
        self._corpus = []
        self._fuzz_drivers = []
        self._treasures = []

        Ticket.__init__(self, 'Fuzz findings', '')
        self._get_resources(sb_tasks)
        self.update_description(self._gen_description())

    @staticmethod
    def _res_cell_description(res):
        if res is None:
            return 'check sb task'
        res_sb_url = Style.url(SB_RES_ID_URL.format(res.id), res.id)
        rev_url = Style.rev_url(res.revision)
        info = '{}\n{}'.format(rev_url, res_sb_url)
        return info

    def get_issue_stats(self):
        total_stats = dict()
        for i, stat in enumerate(self._stats):
            sb_task_id = self._sb_tasks[i].id
            total_stats.update({
                sb_task_id: stat
            })
        return total_stats

    def create(self, session, startrek_url=STARTREK_URL):
        return super(FuzzTicket, self).create(session, startrek_url=startrek_url)

    def _gen_description(self):
        table = Style.Table(
            'Sandbox task id', 'Report revision', 'Build info', 'Index rev/id', 'Corpus rev/id', 'Issues', 'Fuzz time'
        )
        for i, task in enumerate(self._sb_tasks):
            sb_url = Style.url(
                'https://sandbox.yandex-team.ru/task/{}/view'.format(task.id),
                '{}'.format(task.id),
            )
            revision = Style.rev_url(self._revisions[i])
            if revision == 0:
                fuzz_driver_res = get_latest_resource(ReportFuzzExecutable, id=task.id)
                if fuzz_driver_res is None:
                    logger.error('Unable to find fuzz_driver revision')
                else:
                    revision = fuzz_driver_res.revision

            build_info = 'Check sb task'
            if self._fuzz_drivers[i] is None:
                logger.warning('No fuzz driver for task_id={} skipping'.format(task.id))
            else:
                build_info = self._fuzz_drivers[i].description
            index_info = FuzzTicket._res_cell_description(self._index[i])
            corpus_info = FuzzTicket._res_cell_description(self._corpus[i])
            issues = '\n'.join(map(lambda pair: '{}:\t{}'.format(*pair), self._stats[i].items()))
            fuzz_time = self._fuzz_times[i]
            table.row(sb_url, revision, build_info, index_info, corpus_info, issues, fuzz_time)

        return '\n'.join((
            'Some issues were found during last report fuzzing runs.',
            '',
            str(table),
            'Useful links', Style.li(
                Style.url(WIKI_TREASURE_ANALYSIS, 'How to reproduce'),
                Style.url(WIKI_FUZZING_NOTES, 'Notes about fuzzing')
            ),
            '----'
        ))

    def _fill_entry_and_attach(self, task, stats, fuzz_time, revision, index, corpus, fuzz_driver, treasure):
        if task is None:
            logger.error('Task is none!')
            return
        self._sb_tasks.append(task)
        self._stats.append(stats)
        self._fuzz_times.append(fuzz_time)
        self._revisions.append(revision)
        self._index.append(index)
        self.add_resource_attachment(
            'Index_{}.zip'.format(task.id), index
        )
        self._corpus.append(corpus)
        self.add_resource_attachment(
            'Corpus_{}.zip'.format(task.id), index
        )
        self._fuzz_drivers.append(fuzz_driver)
        self._treasures.append(treasure)
        self.add_resource_attachment(
            'Treasures_{}.zip'.format(task.id), treasure
        )

    def _get_resources(self, sb_tasks):
        for task in sb_tasks:
            stats = dict()
            fuzz_runtime = 'Unknown'
            treasure_res = get_latest_resource(ReportFuzzTreasure, task_id=task.id)
            if treasure_res is None:
                logger.warn('No treasures found in task with id={}'.format(task.id))
                self._fill_entry_and_attach(
                    task=task,
                    stats=stats,
                    fuzz_time=fuzz_runtime,
                    revision=int(self._context.Parameters.revision),
                    index=None,
                    corpus=None,
                    fuzz_driver=None,
                    treasure=None
                )
                # Mark processed
                add_tag(task, PROCESSED_TAG)
                continue

            # Process task
            logger.info('Processing resource (id={}) with description "{}"'.format(
                treasure_res.id,
                treasure_res.description
            ))
            revision = treasure_res.revision

            try:
                fuzz_runtime = treasure_res.fuzz_runtime
                stats = json.loads(treasure_res.findings)
            except Exception as e:
                logger.warning('Unable to parse fuzz time and treasure stats due to exception {}'.format(e))
                # TODO: REMOVE - That is legacy
                try:
                    description = json.loads(treasure_res.description)
                    fuzz_runtime = description.get('runtime', 'Unknown')
                    stats = description.get('stats', 'Check treasure')
                except ValueError:
                    logger.warning('Unable to decode json from description')

            fuzz_driver_res = get_latest_resource(ReportFuzzExecutable, task_id=task.id)

            index = get_latest_resource(ReportCollectedIndex, id=treasure_res.index_id)
            if not index:
                index = get_latest_resource(ReportCollectedLiteIndex, id=treasure_res.index_id)
            corpus = get_latest_resource(ReportCollectedCorpus, id=treasure_res.corpus_id)

            self._fill_entry_and_attach(
                task=task,
                stats=stats,
                fuzz_time=fuzz_runtime,
                revision=revision,
                index=index,
                corpus=corpus,
                fuzz_driver=fuzz_driver_res,
                treasure=treasure_res
            )
            # Mark processed
            add_tag(task, PROCESSED_TAG)
